<?php

namespace Inside\Newsletters\Http\Controllers;

use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Illuminate\View\View;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Services\Queries\ContentQuery;
use Inside\Host\Bridge\BridgeContent;
use Inside\Newsletters\Events\NewsletterTestTriggeredEvent;
use Inside\Newsletters\Events\NewsletterTriggeredEvent;
use Inside\Newsletters\Jobs\SendNewsletterJob;
use Inside\Permission\Facades\Role as RoleService;
use Inside\Permission\Models\Role;
use Inside\Settings\Models\Setting;
use Inside\User\Models\User;
use Symfony\Component\Finder\Finder;

/**
 * Notifications controller.
 *
 * @category Class
 * @package  Inside\Newsletters\Http\Controllers\NewslettersController
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class NewslettersController
{
    public function list(Request $request): array
    {
        $config = config('newsletters');
        $types  = implode(',', array_keys($config['types']));

        $service = new ContentQuery();

        $request->merge([
            'fields' => json_encode([
                'uuid',
                'content_type',
                'title',
                'sent_at',
                'created_at',
                'updated_at',
                'published_at',
                'scheduled_at',
                'newsletter_roles',
                'slug',
            ]),
        ]);
        // Don't filter on langcode, newsletter are send in user langcode
        $request = ContentHelper::removeInputFromRequest($request, 'filters', 'langcode');

        $service->requestQuery($types, $request);

        $data = $service->transformCollection();

        /** @var ?Role $authenticatedRole */
        $authenticatedRole = Role::where('name', 'authenticated')->first();

        foreach ($data['data'] as &$newsletter) {
            $roleIDs = json_decode($newsletter['newsletter_roles']);

            if (empty($roleIDs) && $authenticatedRole) {
                $roleIDs = [$authenticatedRole->id];
            }

            $subscribersQuery = User::query()->where('status', 1);
            $subscribersQuery->join('inside_users_roles', 'inside_users.uuid', '=', 'inside_users_roles.user_uuid')
                             ->whereIn(
                                 'inside_users_roles.role_id',
                                 $roleIDs
                             );

            $subscribersQuery->whereNotIn('inside_users.uuid', function ($subQuery) use ($newsletter) {
                $subQuery->select('user_uuid')->from('inside_newsletters_unsubscriptions')->where(
                    'type',
                    $newsletter['content_type']
                );
            });

            $sentQuery = DB::table('inside_newsletters_sent')
                ->join('inside_users', 'inside_users.uuid', '=', 'inside_newsletters_sent.user_uuid')
                ->where('inside_users.status', 1)
                ->where('inside_newsletters_sent.uuid', $newsletter['uuid']);

            $newsletter['newsletter_status'] = [
                'count' => $sentQuery->count(),
                'total' => $sentQuery->count() > 0 ? $sentQuery->count() : $subscribersQuery->get()->unique('uuid')->count(),
            ];

            $newsletter['newsletter_roles'] =
                Role::query()->whereIn('id', $roleIDs)->pluck('name')->transform(function ($name) {
                    return RoleService::getHumanName($name);
                });
        }

        return $data;
    }

    /**
     * Return the html view of a given newsletter
     *
     * @param Request $request
     * @param string  $type
     * @param string  $uuid
     * @return array|\Illuminate\View\View|mixed
     */
    public function view(Request $request, string $type, string $uuid)
    {
        $model = call_user_func(type_to_class($type) . '::find', $uuid);

        if (!$model) {
            abort(404, 'Not found');
        }

        $config = config('newsletters');

        if (!isset($config['types'][$type])) {
            abort(404, 'No view found');
        }

        /** @var User $user */
        $user = Auth::user();

        $data = [
            'to.uuid' => $user->uuid,
            'preview' => true,
        ];

        return view(
            'notifications.' . $config['types'][$type]['notification']['view'],
            ['model' => $model, 'data' => $data]
        );
    }

    /**
     * Send the newsletter (or test with an emails parameter)
     */
    public function send(Request $request, string $type, string $uuid): JsonResponse
    {
        $isEmailSendingEnabled = (bool)Setting::where('key', 'email_sending_enabled')->first()?->value;
        if (!$isEmailSendingEnabled) {
            Log::warning('MailNotifications are disabled - newsletter send skipped ');
            return response()->json([
                'message' => 'MailNotifications are disabled - newsletter send skipped',
                'status' => 'ko',
            ]);
        }
        $model = call_user_func(type_to_class($type) . '::find', $uuid);

        if (!$model) {
            abort(404, 'No model');
        }
        if ($request->has('email')) {
            $emails = $request->get('email');

            NewsletterTestTriggeredEvent::dispatch($model, $emails);
        } else {
            NewsletterTriggeredEvent::dispatch($model);
        }

        return response()->json([
            'status' => 'ok',
        ]);
    }

    /**
     * Get the subscription status of all newsletter types for the current user
     */
    public function getSubscriptions(): JsonResponse
    {
        $config = config('newsletters');
        $types  = $config['types'];

        $data = [];

        /** @var User $user */
        $user = Auth::user();
        /** @var string $langcode */
        $langcode = $user->langcode;
        Lang::setLocale($langcode);

        foreach ($types as $type => $typeConfigs) {
            $restrictSubscription = $typeConfigs['restrict_subscription'] ?? false;
            $roles                = Arr::prepend($typeConfigs['default_roles'] ?? [], 'super_administrator');

            if ($restrictSubscription && !$user->hasAnyRole($roles)) {
                continue;
            }

            $unsubscribed = DB::table('inside_newsletters_unsubscriptions')->where('type', $type)->where('user_uuid', $user->uuid)->exists();
            $data[$type]  = [
                'title'      => Lang::get('newsletter.subscription.' . $type . '.title'),
                'subscribed' => !$unsubscribed,
            ];
        }

        return response()->json(['data' => $data]);
    }

    /**
     * Subscribe to a newsletter type
     */
    public function subscribe(Request $request, string $type): JsonResponse
    {
        /** @var \Inside\Authentication\Models\User $user */
        $user = Auth::user();

        DB::table('inside_newsletters_unsubscriptions')->where('type', $type)->where('user_uuid', $user->uuid)->delete();

        return response()->json([]);
    }

    /**
     * Unsubscribe from a newsletter type
     */
    public function unsubscribe(Request $request, string $type): JsonResponse
    {
        /** @var \Inside\Authentication\Models\User $user */
        $user = Auth::user();

        if (!DB::table('inside_newsletters_unsubscriptions')->where('type', $type)->where('user_uuid', $user->uuid)->exists()) {
            DB::table('inside_newsletters_unsubscriptions')->insert(['type' => $type, 'user_uuid' => $user->uuid]);
        }

        return response()->json([]);
    }

    public function types(): View
    {
        $newsletterDirectory = back_path('resources/newsletters/');
        $finder              = new Finder();
        $finder->files()->in($newsletterDirectory)->name('*.php');

        $newsletters = collect();
        if ($finder->hasResults()) {
            foreach ($finder as $file) {
                $absoluteFilePath      = $file->getRealPath();
                $fileNameWithExtension = $file->getRelativePathname();
                $data                  = require_once $absoluteFilePath;

                $name          = $data['_meta']['name'] ?? $file->getBasename('.' . $file->getExtension());
                $newsletters[] = [
                    'name'   => $name,
                    'type'   => $data['_meta']['type'] ?? $name . '_newsletters',
                    'system' => $file->getBasename('.' . $file->getExtension()),
                    'file'   => str_replace(cms_base_path() . '/', '', $file->getPath()),
                ];
            }
        }

        return view('dashboard.newsletters.list', compact('newsletters'));
    }

    public function editType(string $name): View
    {
        $newsletterConfig = back_path('resources/newsletters/'.$name . '.php');

        if (!File::exists($newsletterConfig)) {
            throw new ModelNotFoundException();
        }
        $config                  = require_once $newsletterConfig;

        return view('dashboard.newsletters.edit', compact('name', 'config'));
    }

    public function update(string $name, Request $request): void
    {
        dd($request->all());
    }

    public function scheduleDelivery(Request $request, string $type, string $uuid): JsonResponse
    {
        $model = type_to_class($type)::find($uuid);

        if (!$model) {
            abort(404, 'No model found !');
        }

        $request->validate(['scheduled_at' => 'required|date']);

        $scheduledDateTime = get_date($request->get('scheduled_at'));

        (new BridgeContent())->contentUpdate($type, [
            'uuid' => $uuid,
            'scheduled_at' => $scheduledDateTime
        ]);

        SendNewsletterJob::dispatch($model)->delay($scheduledDateTime);

        return response()->json(['status' => 'ok']);
    }

    public function cancelScheduledDelivery(Request $request, string $type, string $uuid): JsonResponse
    {
        if (type_to_class($type)::query()->where('uuid', $uuid)->doesntExist()) {
            abort(404, 'No model found !');
        }

        (new BridgeContent())->contentUpdate($type, [
            'uuid' => $uuid,
            'scheduled_at' => null
        ]);

        return response()->json([
            'status' => 'ok',
            'message' => 'Scheduled delivery cancelled',
        ]);
    }
}
