<?php

declare(strict_types=1);

namespace Inside\Newsletters\Http\Controllers;

use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Storage;
use Inside\Kernel\Filesystem\Facades\FileStreamer;
use Inside\Slug\Models\Slug;
use Symfony\Component\HttpFoundation\StreamedResponse;
use Illuminate\Support\Facades\Log;

/**
 * @class NewslettersAssetUrlController
 *
 * Allow streaming newsletter fields assets (e.g. logo, banner)
 */
final class NewslettersAssetUrlController
{
    /**
     * @param string $newsletterUuid
     * @param string $fieldName
     * @return StreamedResponse
     */
    public function serve(string $newsletterUuid, string $fieldName): StreamedResponse
    {
        try {
            $type = str(Slug::where('uuid', $newsletterUuid)->first()?->type)->after('inside_content_')->toString();
            $newsletter = type_to_class($type)::withoutGlobalScopes()->findOrFail($newsletterUuid);
        } catch (ModelNotFoundException $modelNotFoundException) {
            Log::warning("[Serve Newsletter $fieldName] try to access to unknown newsletters $fieldName uuid ($newsletterUuid)");
            throw $modelNotFoundException;
        }

        $filePath = $newsletter->{$fieldName};

        $disk = Storage::disk('local');

        if (! $disk->exists($filePath)) {
            Log::warning("[Serve Newsletter $fieldName] try to access to unknown file $filePath for $newsletterUuid");
            throw new ModelNotFoundException();
        }

        return FileStreamer::streamFile($filePath);
    }
}
