<?php

namespace Inside\Newsletters\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Inside\Content\Facades\DynamicClass;
use Inside\Content\Facades\Schema;
use Inside\Host\Bridge\BridgeContentType;
use Inside\I18n\Models\Language;
use Inside\Support\Str;

class SyncNewsletterCommand extends Command
{
    protected $signature = 'inside:newsletter:sync {name}';

    protected $description = 'Synchronize a newsletter with the system';

    /** @var array */
    protected $fields = [];

    /** @var array */
    protected $languages = [];

    public function handle(): void
    {
        DynamicClass::disableRebuildAutoLoad();
        /** @var string $name */
        $name = $this->argument('name');
        $name = trim(Str::lower($name));
        $this->writeln('<fg=magenta>Synchronisation de la newsletter <fg=cyan>'
            .$name.'</> à partir de sa configuration</fg=magenta>');

        $newsletterConfig = back_path('resources/newsletters/'.$name.'.php');

        if (!File::exists($newsletterConfig)) {
            $this->error('Impossible de charger la configuration de la newsletter <fg=cyan>'
                .$name.'</>');
            return;
        }
        $this->write('<fg=green>Chargement de la configuration</>');
        try {
            $config = require_once $newsletterConfig;
        } catch (\Throwable $e) {
            $this->writeResult(false);
            $this->warn($e->getMessage());
            return;
        }
        $this->writeResult(true);

        $newsletterType = $config['_meta']['type'] ?? 'newsletter_'.$name;
        $newsletterName = $config['_meta']['name'] ?? 'Newsletter';
        $fieldsPrefix = $config['_meta']['fieldsPrefix'] ?? 'iln'.$name[0].'_';

        // Prepare locales
        foreach (list_languages() as $locale) {
            $this->languages[$locale] = Language::where('locale', $locale)
                ->first();
        }
        $isMultiLanguages = count($this->languages) > 1;

        if ($isMultiLanguages) {
            $this->writeln('<fg=yellow>Détection d\'une instance multi-lingue</>');
        }

        // Prepare content type
        $this->checkAndPrepareNewsletterContentType($newsletterType, $newsletterName);

        // Prepare general fields
        $this->checkAndPrepareNewsletterFields($newsletterType);

        // Fieldsets
        $this->prepareGroupIfNecessary($newsletterType);

        // -- Banner part

        // Prepare title & subtitle if necessary
        $bannerUseTitleField = $config['banner']['useTitleField'] ?? false;
        if ($bannerUseTitleField) {
            $titleFieldName = $config['banner']['titleFieldName'] ?? 'banner_title';
            $subTitleFieldName = $config['banner']['subTitleFieldName'] ?? 'banner_subtitle';

            foreach ($this->languages as $locale => $language) {
                $this->checkAndPrepareNewsletterBannerTitleAndSubtitleFields(
                    $newsletterType,
                    $titleFieldName.($isMultiLanguages ? "_$locale" : ''),
                    $subTitleFieldName.($isMultiLanguages ? "_$locale" : ''),
                    ' - '.$language->name
                );
            }
        }

        // Prepare image field if necessary
        $bannerUseTitleField = $config['banner']['useImageField'] ?? false;
        if ($bannerUseTitleField) {
            $bannerImageFieldName = $config['banner']['imageFieldName'] ?? 'banner_image';
            foreach ($this->languages as $locale => $language) {
                $this->checkAndPrepareNewsletterBannerImageField(
                    $newsletterType,
                    $bannerImageFieldName.($isMultiLanguages ? "_$locale" : ''),
                    ' - '.$language->name
                );
            }
        }

        // -- Introduction

        // Prepare Introduction field
        if (isset($config['introduction'])) {
            foreach ($this->languages as $locale => $language) {
                $introductionFieldName
                    = $config['introduction']['introductionFieldName'] ??
                    'newsletter_introduction';
                $this->checkAndPrepareNewsletterIntroductionField(
                    $newsletterType,
                    $introductionFieldName.($isMultiLanguages ? "_$locale"
                        : ''),
                    ' - '.$language->name
                );
            }
        }

        // -- sections
        if (isset($config['sections']) && !empty($config['sections'])) {
            $this->checkAndPrepareNewsletterSection($newsletterType, $config['sections'], $fieldsPrefix);
        }

        // -- newsletters
        $this->cleanupAndFinishNewsletter($newsletterType);

        DynamicClass::enableRebuildAutoLoad();
        Schema::refresh();
        $this->call('models:rebuild');
    }

    /**
     * @param  string  $type
     * @param  string  $label
     *
     * @throws \Inside\Host\Exceptions\TableAlreadyExistException
     */
    protected function checkAndPrepareNewsletterContentType(string $type, string $label): void
    {
        if (Schema::hasContentType($type)) {
            return; // Type is set and exists ! cool
        }

        $this->write('<fg=yellow>Le type de contenu <fg=cyan>'.$type.'</fg=cyan> pour cette newsletter n\'existe pas et va être créé.</>');
        Log::warning('[sync newsletter] newsletter content type['.$type.'] does not exist, creating ...');
        $bridge = new BridgeContentType();

        $titles = [];
        foreach (list_languages() as $locale) {
            $titles[$locale] = $label; // TODO: may be customize here
        }

        try {
            $bridge->contentTypeCreate($type, [
                'translatable' => false,
                'searchable' => false,
                'global_searchable' => false,
                'aliasable' => true,
                'permissible' => true,
                'categorizable' => false,
                'title' => $titles,
            ]);
        } catch (\Throwable $e) {
            $this->writeResult(false);
            return;
        }
        $this->writeResult(true);
    }

    /**
     * @param  string  $type
     *
     * @throws \Inside\Host\Exceptions\TableAlreadyExistException
     */
    protected function checkAndPrepareNewsletterFields(string $type): void
    {
        $bridge = new BridgeContentType();

        if (! Schema::hasField($type, 'newsletter_roles')) {
            $this->write('<fg=yellow>Le champ <fg=cyan>newsletter_roles</fg=cyan> pour cette newsletter n\'existe pas et va être créé.</>');
            Log::warning('[sync newsletter] newsletter field newsletter_roles does not exist, creating ...');

            $bridge->contentTypeCreateField($type, 'newsletter_roles', [
                'name' => 'newsletter_roles',
                'searchable' => false,
                'translatable' => false,
                'default' => null,
                'search_result_field' => false,
                'required' => false,
                'weight' => 10000,
                'title' => [
                    'fr' => 'Roles newsletter',
                    'en' => 'Newsletter Roles',
                    'de' => 'Newsletter Rollen',
                    'es' => 'Roles del boletín',
                    'pt' => 'Papéis da newsletter',
                    'nl' => 'Nieuwsbriefrollen',
                ],
                'description' => [],
                'widget' => [
                    'type' => BridgeContentType::STRING_WIDGET,
                    'hidden' => true,
                    'settings' => [
                        'size' => 240,
                        'placeholder' => '',
                    ],
                ],
                'type' => BridgeContentType::TEXT_FIELD,
                'settings' => [
                    'max_length' => 255,
                    'cardinality' => 1,
                ],
            ]);

            $this->writeResult(true);
        }

        if (! Schema::hasField($type, 'scheduled_at')) {
            $this->write('<fg=yellow>Le champ <fg=cyan>scheduled_at</fg=cyan> pour cette newsletter n\'existe pas et va être créé.</>');
            Log::warning('[sync newsletter] newsletter field scheduled_at does not exist, creating ...');

            $bridge->contentTypeCreateField($type, 'scheduled_at', [
                'name' => 'scheduled_at',
                'searchable' => false,
                'translatable' => false,
                'search_result_field' => false,
                'required' => false,
                'weight' => 0,
                'title' => [
                    'fr' => 'Programmé à',
                    'en' => 'Scheduled at',
                    'es' => 'Programado en',
                    'de' => 'Geplant am',
                    'pt' => 'Programado em',
                    'nl' => 'Gepland op',
                ],
                'description' => [],
                'widget' => [
                    'hidden' => true,
                ],
                'type' => BridgeContentType::TIMESTAMP_FIELD,
                'settings' => [
                    'cardinality' => 1,
                ],
            ]);

            $this->writeResult(true);
        }

        if (! Schema::hasField($type, 'sent_at')) {
            $this->write('<fg=yellow>Le champ <fg=cyan>sent_at</fg=cyan> pour cette newsletter n\'existe pas et va être créé.</>');
            Log::warning('[sync newsletter] newsletter field sent_at does not exist, creating ...');

            $bridge->contentTypeCreateField($type, 'sent_at', [
                'name' => 'sent_at',
                'searchable' => false,
                'translatable' => false,
                'search_result_field' => false,
                'required' => false,
                'weight' => 0,
                'title' => [
                    'fr' => 'Envoyé le',
                    'en' => 'Sent at',
                    'es' => 'Enviado el',
                    'de' => 'Gesendet am',
                    'pt' => 'Enviado em',
                    'nl' => 'Verzonden op',
                ],
                'description' => [],
                'widget' => [
                    'hidden' => true,
                ],
                'type' => BridgeContentType::TIMESTAMP_FIELD,
                'settings' => [
                    'cardinality' => 1,
                ],
            ]);

            $this->writeResult(true);
        }

        if (! Schema::hasField($type, 'nl_logo')) {
            $this->write('<fg=yellow>Le champ <fg=cyan>nl_logo</fg=cyan> pour cette newsletter n\'existe pas et va être créé.</>');
            Log::warning('[sync newsletter] newsletter field nl_logo does not exist, creating ...');

            $bridge->contentTypeCreateField($type, 'nl_logo', [
                'name' => 'nl_logo',
                'searchable' => false,
                'translatable' => true,
                'search_result_field' => false,
                'searchable_filter' => false,
                'required' => false,
                'weight' => 1,
                'title' => [
                    'fr' => 'Logo',
                    'en' => 'Logo',
                    'de' => 'Logo',
                    'es' => 'Logotipo',
                    'pt' => 'Logotipo',
                    'nl' => 'Logo',
                ],
                'description' => [
                    'fr' => "L’image par défaut sera l’image présente dans les paramètres avancés de l’intranet. Si vous ajoutez une autre image ici elle écrasera l’image des paramètres avancés sur cette newsletter",
                    'en' => "The default image will be the one present in the advanced settings of the intranet. If you add another image here, it will override the image from the advanced settings on this newsletter.",
                    'de' => "Das Standardbild ist das im erweiterten Intranet-Setup. Wenn Sie hier ein anderes Bild hinzufügen, wird das Bild der erweiterten Einstellungen in diesem Newsletter überschrieben.",
                    'es' => "La imagen predeterminada será la que esté presente en la configuración avanzada del intranet. Si añades otra imagen aquí, sobrescribirá la imagen de la configuración avanzada en este boletín.",
                    'pt' => "A imagem padrão será a presente nas configurações avançadas da intranet. Se adicionar outra imagem aqui, ela substituirá a imagem das configurações avançadas nesta newsletter.",
                    'nl' => "De standaardafbeelding is de afbeelding die aanwezig is in de geavanceerde instellingen van het intranet. Als u hier een andere afbeelding toevoegt, overschrijft deze de afbeelding uit de geavanceerde instellingen op deze nieuwsbrief."
                ],
                'widget' => [
                    'type' => BridgeContentType::IMAGE_WIDGET,
                    'settings' => [
                        'progress_indicator' => 'throbber',
                        'preview_image_style' => 'thumbnail',
                    ],
                    'weight' => 1,
                ],
                'type' => BridgeContentType::IMAGE_FIELD,
                'settings' => [
                    'uri_scheme' => 'public',
                    'default_image' => [
                        'uuid' => '',
                        'alt' => '',
                        'title' => '',
                        'width' => null,
                        'height' => null,
                    ],
                    'target_type' => 'file',
                    'display_field' => false,
                    'display_default' => false,
                    'cardinality' => 1,
                ]
            ]);

            $this->writeResult(true);
        }

        if (! Schema::hasField($type, 'nl_banner')) {
            $this->write('<fg=yellow>Le champ <fg=cyan>nl_banner</fg=cyan> pour cette newsletter n\'existe pas et va être créé.</>');
            Log::warning('[sync newsletter] newsletter field nl_banner does not exist, creating ...');

            $bridge->contentTypeCreateField($type, 'nl_banner', [
                'name' => 'nl_banner',
                'searchable' => false,
                'translatable' => true,
                'search_result_field' => false,
                'searchable_filter' => false,
                'required' => false,
                'weight' => 1,
                'title' => [
                    'fr' => 'Bannière',
                    'en' => 'Banner',
                    'de' => 'Banner',
                    'es' => 'Pancarta',
                    'pt' => 'Banner',
                    'nl' => 'Banner',
                ],
                'description' => [
                    'fr' => "L’image par défaut sera l’image présente dans les paramètres avancés de l’intranet. Si vous ajoutez une autre image ici elle écrasera l’image des paramètres avancés sur cette newsletter",
                    'en' => "The default image will be the one present in the advanced settings of the intranet. If you add another image here, it will override the image from the advanced settings on this newsletter.",
                    'de' => "Das Standardbild ist das im erweiterten Intranet-Setup. Wenn Sie hier ein anderes Bild hinzufügen, wird das Bild der erweiterten Einstellungen in diesem Newsletter überschrieben.",
                    'es' => "La imagen predeterminada será la que esté presente en la configuración avanzada del intranet. Si añades otra imagen aquí, sobrescribirá la imagen de la configuración avanzada en este boletín.",
                    'pt' => "A imagem padrão será a presente nas configurações avançadas da intranet. Se adicionar outra imagem aqui, ela substituirá a imagem das configurações avançadas nesta newsletter.",
                    'nl' => "De standaardafbeelding is de afbeelding die aanwezig is in de geavanceerde instellingen van het intranet. Als u hier een andere afbeelding toevoegt, overschrijft deze de afbeelding uit de geavanceerde instellingen op deze nieuwsbrief."
                ],
                'widget' => [
                    'type' => BridgeContentType::IMAGE_WIDGET,
                    'settings' => [
                        'progress_indicator' => 'throbber',
                        'preview_image_style' => 'thumbnail',
                    ],
                    'weight' => 1,
                ],
                'type' => BridgeContentType::IMAGE_FIELD,
                'settings' => [
                    'uri_scheme' => 'public',
                    'default_image' => [
                        'uuid' => '',
                        'alt' => '',
                        'title' => '',
                        'width' => null,
                        'height' => null,
                    ],
                    'target_type' => 'file',
                    'display_field' => false,
                    'display_default' => false,
                    'cardinality' => 1,
                ]
            ]);

            $this->writeResult(true);
        }
    }

    /**
     * @param  array  $sections
     * @param  string  $fieldsPrefix
     * @throws \Drupal\Core\Entity\EntityStorageException
     * @throws \Inside\Host\Exceptions\ColumnAlreadyExistsException
     */
    protected function checkAndPrepareNewsletterSection(string $newsletterType, array $sections, string $fieldsPrefix = ''): void
    {
        $i = 0;
        foreach ($sections as $sectionName => $sectionConfig) {
            if ($sectionName === 'activities') {
                continue;
            }

            $types = $sectionConfig['fields'] ?? ($sectionConfig['types'] ?? ['news']);
            foreach ($types as $key => $type) {
                $fieldName = is_string($key) ? $key : $type;
                $this->checkAndPrepareNewsletterReferenceField($newsletterType, $fieldsPrefix.$fieldName, $type, $sectionConfig, $i);
            }
        }
    }

    /**
     * @param  string  $type
     * @param  string  $titleFieldName
     * @param  string  $subTitleFieldName
     *
     * @throws \Drupal\Core\Entity\EntityStorageException
     * @throws \Inside\Host\Exceptions\ColumnAlreadyExistsException
     */
    protected function checkAndPrepareNewsletterBannerTitleAndSubtitleFields(string $type, string $titleFieldName, string $subTitleFieldName, string $suffix = ''): void
    {
        if (!Schema::hasContentType($type)) {
            return; // Type is set and exists ! cool
        }
        $this->fields[] = $titleFieldName;
        $this->fields[] = $subTitleFieldName;
        $bridge = new BridgeContentType();
        if (Schema::hasField($type, $titleFieldName)) {
            $this->warn('Le champ <fg=cyan>'.$titleFieldName
                .'</> existe déjà et sera mis-à-jour');
        }

        $this->write('<fg=green>Paramètrage du champ  <fg=cyan>'.$titleFieldName
            .'</></>');

        try {
            $bridge->contentTypeCreateOrUpdateField($type, $titleFieldName, [
                'searchable' => false,
                'translatable' => false,
                'default' => '',
                'search_result_field' => false,
                'required' => false,
                'title' => [
                    'fr' => 'Titre ( Bannière )' . $suffix,
                    'en' => 'Title ( Banner )' . $suffix,
                    'es' => 'Título ( Banner )' . $suffix,
                    'de' => 'Titel ( Banner )' . $suffix,
                    'nl' => 'Titel ( Banner )' . $suffix,
                    'pt' => 'Título ( Banner )' . $suffix,
                ],
                'description' => [],
                'widget' => [
                    'type' => BridgeContentType::TEXT_WIDGET,
                    'settings' => [],
                ],
                'type' => BridgeContentType::TEXT_FIELD,
                'settings' => [
                    'cardinality' => 1,
                ],
            ]);
        } catch (\Throwable $e) {
            $this->writeResult(false);
            $this->warn($e->getMessage());
            return;
        }
        $this->writeResult(true);

        if (Schema::hasField($type, $subTitleFieldName)) {
            $this->warn('Le champ <fg=cyan>'.$subTitleFieldName
                .'</> existe déjà et sera mis-à-jour');
        }

        $this->write('<fg=green>Paramètrage du champ  <fg=cyan>'
            .$subTitleFieldName.'</></>');

        try {
            $bridge->contentTypeCreateOrUpdateField($type, $subTitleFieldName, [
                'searchable' => false,
                'translatable' => false,
                'default' => '',
                'search_result_field' => false,
                'required' => false,
                'title' => [
                    'fr' => 'Sous-titre ( Bannière )' . $suffix,
                    'en' => 'Subtitle ( Banner )' . $suffix,
                    'es' => 'Subtítulo ( Banner )' . $suffix,
                    'de' => 'Untertitel ( Banner )' . $suffix,
                    'nl' => 'Ondertitel ( Banner )' . $suffix,
                    'pt' => 'Subtítulo ( Banner )' . $suffix,
                ],
                'description' => [],
                'widget' => [
                    'type' => BridgeContentType::TEXT_WIDGET,
                    'settings' => [],
                ],
                'type' => BridgeContentType::TEXT_FIELD,
                'settings' => [
                    'cardinality' => 1,
                ],
            ]);
        } catch (\Throwable $e) {
            $this->writeResult(false);
            $this->warn($e->getMessage());
            return;
        }
        $this->writeResult(true);
    }

    /**
     * @param  string  $type
     * @param  string  $imageFieldName
     *
     * @throws \Drupal\Core\Entity\EntityStorageException
     * @throws \Inside\Host\Exceptions\ColumnAlreadyExistsException
     */
    protected function checkAndPrepareNewsletterBannerImageField(string $type, string $imageFieldName, string $suffix = ''): void
    {
        if (!Schema::hasContentType($type)) {
            return; // Type is set and exists ! cool
        }
        $this->fields[] = $imageFieldName;
        $bridge = new BridgeContentType();
        if (Schema::hasField($type, $imageFieldName)) {
            $this->warn('Le champ <fg=cyan>'.$imageFieldName
                .'</> existe déjà et sera mis-à-jour');
        }

        $this->write('<fg=green>Paramètrage du champ  <fg=cyan>'.$imageFieldName
            .'</></>');

        try {
            $bridge->contentTypeCreateOrUpdateField($type, $imageFieldName, [
                'searchable' => false,
                'translatable' => false,
                'default' => null,
                'search_result_field' => false,
                'required' => false,
                'title' => [
                    'fr' => 'Image ( Bannière )' . $suffix,
                    'en' => 'Image ( Banner )' . $suffix,
                    'es' => 'Imagen ( Banner )' . $suffix,
                    'de' => 'Bild ( Banner )' . $suffix,
                    'nl' => 'Afbeelding ( Banner )' . $suffix,
                    'pt' => 'Imagem ( Banner )' . $suffix,
                ],
                'description' => [],
                'widget' => [
                    'type' => BridgeContentType::IMAGE_WIDGET,
                    'settings' => [],
                ],
                'type' => BridgeContentType::IMAGE_FIELD,
                'settings' => [
                    'cardinality' => 1,
                ],
            ]);
        } catch (\Throwable $e) {
            $this->writeResult(false);
            $this->warn($e->getMessage());
            return;
        }
        $this->writeResult(true);
    }

    /**
     * @param  string  $type
     * @param  string  $introductionFieldName
     * @param  string  $suffix
     *
     * @throws \Drupal\Core\Entity\EntityStorageException
     * @throws \Inside\Host\Exceptions\ColumnAlreadyExistsException
     */
    protected function checkAndPrepareNewsletterIntroductionField(string $type, string $introductionFieldName, string $suffix = ''): void
    {
        if (!Schema::hasContentType($type)) {
            return; // Type is set and exists ! cool
        }
        $this->fields[] = $introductionFieldName;
        $bridge = new BridgeContentType();
        if (Schema::hasField($type, $introductionFieldName)) {
            $this->warn('Le champ <fg=cyan>'.$introductionFieldName
                .'</> existe déjà et sera mis-à-jour');
        }

        $this->write('<fg=green>Paramètrage du champ  <fg=cyan>'
            .$introductionFieldName.'</></>');

        try {
            $bridge->contentTypeDeleteField('node', $type, $introductionFieldName);
            $bridge->contentTypeCreateOrUpdateField(
                $type,
                $introductionFieldName,
                [
                    'searchable' => false,
                    'translatable' => false,
                    'default' => '',
                    'search_result_field' => false,
                    'required' => false,
                    'title' => [
                        'fr' => 'Introduction' . $suffix,
                        'en' => 'Introduction' . $suffix,
                        'es' => 'Introducción' . $suffix,
                        'de' => 'Einleitung' . $suffix,
                        'nl' => 'Inleiding' . $suffix,
                        'pt' => 'Introdução' . $suffix,
                    ],
                    'description' => [],
                    'widget' => [
                        'type' => BridgeContentType::LONGTEXT_WIDGET,
                        'settings' => [
                            'rows' => 9,
                            'summary_rows' => 3,
                            'placeholder' => '',
                        ],
                    ],
                    'type' => BridgeContentType::TEXTAREA_FIELD,
                    'settings' => [
                        'cardinality' => 1,
                    ],
                ]
            );
        } catch (\Throwable $e) {
            $this->writeResult(false);
            $this->warn($e->getMessage());
            return;
        }
        $this->writeResult(true);
    }

    /**
     * @param  string  $type
     * @param  string  $referenceFieldName
     * @param string $referenceType
     * @param int $order
     *
     * @throws \Drupal\Core\Entity\EntityStorageException
     * @throws \Inside\Host\Exceptions\ColumnAlreadyExistsException
     */
    protected function checkAndPrepareNewsletterReferenceField(string $type, string $referenceFieldName, string $referenceType, array $sectionConfig, int $order = 0): void
    {
        if (!Schema::hasContentType($type)) {
            return; // Type is set and exists ! cool
        }
        $this->fields[] = $referenceFieldName;
        $bridge = new BridgeContentType();

        $referenceOptions = Schema::getModelOptions($referenceType);

        if (Schema::hasField($type, $referenceFieldName)) {
            $this->warn('Le champ <fg=cyan>'.$referenceFieldName.'</> existe déjà et sera mis-à-jour');
        }

        $this->write('<fg=green>Paramètrage du champ  <fg=cyan>'.$referenceFieldName.'</></>');

        try {
            $bridge->contentTypeCreateOrUpdateField(
                $type,
                $referenceFieldName,
                [
                    'searchable' => false,
                    'translatable' => false,
                    'default' => [],
                    'search_result_field' => false,
                    'required' => false,
                    'weight' => (500 + $order),
                    'title' => [
                        'fr' => $sectionConfig['fieldLabel'][$referenceFieldName]['fr'] ?? ($referenceOptions['title']['fr'] ?? 'Contenu'),
                        'en' => $sectionConfig['fieldLabel'][$referenceFieldName]['en'] ?? ($referenceOptions['title']['en'] ?? 'Content'),
                        'es' => $sectionConfig['fieldLabel'][$referenceFieldName]['es'] ?? ($referenceOptions['title']['es'] ?? 'Contenido'),
                        'de' => $sectionConfig['fieldLabel'][$referenceFieldName]['de'] ?? ($referenceOptions['title']['de'] ?? 'Inhalt'),
                        'nl' => $sectionConfig['fieldLabel'][$referenceFieldName]['nl'] ?? ($referenceOptions['title']['nl'] ?? 'Inhoud'),
                        'pt' => $sectionConfig['fieldLabel'][$referenceFieldName]['pt'] ?? ($referenceOptions['title']['pt'] ?? 'Conteúdo'),
                    ],
                    'description' => [],
                    'widget' => [
                        'type' => BridgeContentType::AUTOCOMPLETE_WIDGET,
                        'group' => 'newsletter_contents',
                        'settings' => [],
                        'weight' => (500 + $order),
                    ],
                    'type' => BridgeContentType::REFERENCE_FIELD,
                    'settings' => [
                        'target_type' => [$referenceType],
                        'cardinality' => -1,
                    ],
                ],
                $order
            );
        } catch (\Throwable $e) {
            $this->writeResult(false);
            $this->warn($e->getMessage());
            return;
        }
        $this->writeResult(true);
    }

    protected function prepareGroupIfNecessary(string $type): void
    {
        $bridge = new BridgeContentType();

        $labels = [];
        foreach (['fr','en','es','de','nl','pt'] as $locale) {
            $labels[$locale] = trans(
                'newsletter.contentsfieldset',
                [],
                $locale
            );
        }
        $this->write('<fg=green>Préparation du fieldset des contenus de newsletter</>');

        try {
            $bridge->createOrUpdateFieldGroup($type, 'newsletter_contents', $labels, 500);
        } catch (\Throwable $e) {
            $this->writeResult(false);
            $this->warn($e->getMessage());
            return;
        }
        $this->writeResult(true);
    }

    /**
     * @param  string  $type
     */
    protected function cleanupAndFinishNewsletter(string $type): void
    {
        $bridge = new BridgeContentType();
        $this->write('<fg=green>Nettoyage de la newsletter, changement de l\'ordre des champs, et suppression des champs non nécessaires au formulaire</>');

        try {
            // Reorder options fields
            $this->reorderFieldsGroup($type, 'group_options', 1500);
            $this->reorderFieldsGroup($type, 'group_advanced_options', 2000);

            $bridge->changeFieldFormOptions($type, 'newsletter_roles', false);
            $bridge->changeFieldFormOptions($type, 'langcode', false);
            $bridge->changeFieldFormOptions($type, 'updated_at', false);
            $bridge->changeFieldFormOptions($type, 'status', false);

            $bridge->changeFieldFormOptions($type, 'created_at', true, 'group_options');
            $bridge->changeFieldFormOptions($type, 'authors', true, 'group_options');

            // Reorder form fields
            $bridge->reorderFields($type, array_merge(['title'], $this->fields, ['status', 'authors', 'create_at']));
        } catch (\Throwable $e) {
            $this->writeResult(false);
            $this->warn($e->getMessage());
            return;
        }
        $this->writeResult(true);
        $fieldNames = Schema::getFieldListing($type);
        $removedFields = array_diff($fieldNames, array_merge([
            'authors',
            'created_at',
            'langcode',
            'status',
            'title',
            'updated_at',
            'newsletter_roles',
            'scheduled_at',
            'sent_at',
            'nl_banner',
            'nl_logo'
        ], $this->fields));
        if (!empty($removedFields)) {
            $this->warn('Certains champs semblent ne plus exister, il faudrait le controller.');
            $this->table(['fieldnames'], array_map(function ($fieldName) {
                return [$fieldName];
            }, $removedFields));
        }
    }

    /**
     * Write
     *
     * @param  string  $message
     */
    protected function write(string $message): void
    {
        $this->getOutput()->write($message);
    }

    /**
     * Write and return to new line
     *
     * @param  string  $message
     */
    protected function writeln(string $message): void
    {
        $this->getOutput()->writeln($message);
    }

    /**
     * Write a result
     *
     * @param  bool  $success
     */
    protected function writeResult(bool $success): void
    {
        if ($success) {
            $this->writeln(' <fg=green>✔</fg=green>');

            return;
        }
        $this->writeln(' <fg=red>✘</fg=red>');
    }

    /**
     * @param  string  $type
     * @param  string  $name
     * @param  int  $weight
     */
    protected function reorderFieldsGroup(string $type, string $name, int $weight): void
    {
        $groupInfos = field_group_load_field_group(
            $name,
            'node',
            $type,
            'form',
            'default'
        );

        $groupInfos->weight = $weight;

        field_group_group_save((object) $groupInfos);
    }
}
