<?php

namespace Inside\Newsletters\Console\Commands;

use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Inside\Console\Command;
use Inside\Newsletters\Models\NewslettersStatistics as OldStatistics;

class MigrateOldNewslettersStats extends Command
{
    /**
     * @var string
     */
    protected $signature = 'inside:stats:newsletters:migrate {--force}';

    /**
     * @var string
     */
    protected $description = 'Migrate old stats data to new statistics tables.';

    /**
     * @return void
     */
    public function handle(): void
    {
        if (! $this->option('force') && ! $this->confirm('The command must be executed once. Are you sure to continue with the execution? [It is recommended to download the database backup before execution.]')) {
            $this->info('Execution aborted !');
            exit;
        }

        $this->info('Migrate Newsletters stats ...');

        $newslettersConfig = config('newsletters.types', []);
        $newslettersTypes = collect($newslettersConfig)->keys();

        $newslettersTypes->each(function ($type) {
            $statsTableName = type_to_stats_table(class_to_type($type));

            if ($statsTableName === null || ! Schema::hasTable($statsTableName)) {
                $this->error("[Migrate-Newsletters-Statistics] Table '{$statsTableName}' does not exist.");
                exit;
            }
        });

        $newslettersTypes->each(function ($type) {
            type_to_class($type)::select('*')->each(function ($newsletter) use ($type) {
                OldStatistics::where('newsletters_uuid', $newsletter->uuid)->each(function ($oldStat) use ($type) {
                    $statsTableName = type_to_stats_table($type);

                    if ($statsTableName === null) {
                        throw new \Exception('Provided type not valid !');
                    }

                    DB::table($statsTableName)->insert([
                        'user_uuid'         => $oldStat->user_uuid,
                        'content_uuid'      => $oldStat->newsletters_uuid,
                        'created_at'        => get_date($oldStat->created_at)?->format('Y-m-d H:i:s'),
                        'statistic_type'    => $oldStat->statistic_type,
                    ]);
                });
            });
        });

        $this->info('Migrate Newsletters stats have been terminated!');
    }
}
