<?php

namespace Inside\Newsletters\Console\Commands;

use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Inside\Console\Command;
use Inside\Content\Facades\Schema as InsideSchema;
use Illuminate\Support\Facades\DB;

class CreateNewslettersStatsTables extends Command
{
    /**
     * @var string
     */
    protected $signature = 'inside:stats:newsletters:tables';

    /**
     * @var string
     */
    protected $description = 'Read newsletters config and create required tables';

    /**
     * @return void
     */
    public function handle(): void
    {
        $this->info('Creating newsletters stats tables started...');
        $newslettersTypes = config('newsletters.types', []);

        if (! $newslettersTypes) {
            $this->error('Can\'t create stats tables there is no types configured !');

            return;
        }

        collect($newslettersTypes)->keys()->each(function ($type) {
            $this->createStatsTable($type);
        });
    }

    private function createStatsTable(string $type): void
    {
        if (! InsideSchema::hasContentType($type)) {
            $this->error('Content type : '.$type.' does not exist !');

            return;
        }
        $statsTableName = type_to_stats_table($type);
        if ($statsTableName === null) {
            $this->error('Stats table name not found !');

            return;
        }
        if (Schema::hasTable($statsTableName)) {
            $this->warning('Table :'.$statsTableName.' already exists !');

            return;
        }
        try {
            Schema::create($statsTableName, function (Blueprint $table) use ($type) {
                $table->bigIncrements('id');
                $table->uuid('user_uuid');
                $table->uuid('content_uuid');
                $table->string('statistic_type');
                $table->foreign('user_uuid')->references('uuid')->on('inside_content_users')->onDelete('cascade');
                $table->foreign('content_uuid')->references('uuid')->on(type_to_table($type))->onDelete('cascade');
                $table->timestamp('created_at')->useCurrent();
            });
            $this->info('Stats table '.$statsTableName.' of type : '.$type.' has been created !');
        } catch (\Exception $e) {
            Schema::dropIfExists($statsTableName);
            $this->error('Error occurred while trying to create state table : '.$e->getMessage());
        }
    }
}
