<?php

declare(strict_types=1);

namespace Inside\Movements\Services;

use Inside\Content\Models\Contents\Movements;
use Inside\Content\Models\Contents\MovementsCategories;
use Inside\Content\Models\Contents\Users;
use Inside\Database\Eloquent\Builder;
use Inside\Host\Bridge\BridgeContent;
use Inside\Movements\Contracts\UsersMovementsGenerator as UsersMovementsGeneratorContracts;

final class UsersMovementsGenerator implements UsersMovementsGeneratorContracts
{
    private bool $isAutoCreationEnabled;

    private ?BridgeContent $bridge = null;

    public function __construct()
    {
        $this->isAutoCreationEnabled = config('movements.auto_creation_enabled');
    }

    private function getBridge(): BridgeContent
    {
        return $this->bridge ??= new BridgeContent();
    }

    public function autoGenerateUserMovements(Users $user, string $template = 'arrival', bool $force = false): void
    {
        if (! $this->isAutoCreationEnabled && ! $force) {
            return;
        }

        $locale = config('app.locale');

        $category = MovementsCategories::where('template', $template)->where('langcode', $locale)->first();

        if (! $category instanceof MovementsCategories) {
            throw new \LogicException('Invalid movements category template');
        }

        $alreadyExists = Movements::query()
            ->whereHas('movementsUsers', fn (Builder $query) => $query->where('uuid', $user->uuid))
            ->whereHas('movementsCategories', fn (Builder $query) => $query->where('template', $template))
            ->where('langcode', $locale)
            ->exists();

        if ($alreadyExists) {
            return;
        }

        $original = $this->createMovement($user, $category, $locale);
        $original = Movements::findOrFail($original);

        collect(list_languages())
            ->reject($locale)
            ->each(fn (string $langcode) => $this->createMovement($user, $category, $langcode, $original));
    }

    private function createMovement(Users $user, MovementsCategories $category, string $langcode, ?Movements $original = null): ?string
    {
        $category = $category->getTranslationIfExists($langcode);

        if ($category->langcode !== $langcode) {
            return null;
        }

        $data = [
            'movements_categories' => $category->uuid,
            'movements_users' => $user->uuid,
            'langcode' => $langcode,
            'title' => $user->name,
        ];

        if ($original instanceof Movements) {
            $data['uuid_host'] = $original->uuid_host;
        }

        return $this->getBridge()->contentInsert('movements', $data);
    }
}
