<?php

namespace Inside\Groups\Services;

use Exception;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\QueryException;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inside\Authentication\Models\User;
use Inside\Content\Events\UserMentionedInCommentEvent;
use Inside\Content\Exceptions\ModelSchemaNotFoundException;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Contents\Groups;
use Inside\Content\Models\Contents\GroupsDocuments;
use Inside\Content\Models\Contents\GroupsPosts;
use Inside\Content\Models\Contents\Users;
use Inside\Groups\Events\GroupsDocumentsCreatedEvent;
use Inside\Groups\Events\GroupsPostsAnsweredEvent;
use Inside\Groups\Events\GroupsPostsCreatedEvent;
use Inside\Groups\Events\UserRevokedAdminEvent;
use Inside\Groups\Models\Group;
use Inside\Groups\Models\GroupActivity;
use Inside\Groups\Models\GroupPost;
use Inside\Host\Bridge\BridgeContent;
use Inside\Notify\Events\CustomNotificationEvent;
use Inside\Notify\Facades\NotificationType as NotificationTypeHelper;
use Inside\Notify\Models\NotificationSubscriber;
use Inside\Notify\Models\NotificationType;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Facades\Role;
use Inside\Permission\Models\Role as InsideRole;
use InvalidArgumentException;

class GroupsService
{
    protected array $groupTypes = [
        'groups_posts',
        'groups_folders',
        'groups_documents',
    ];

    protected array $roles = [];

    private ?BridgeContent $bridge = null;

    private function getBridge(): BridgeContent
    {
        return $this->bridge ??= new BridgeContent();
    }

    public function touch(Groups $group): void
    {
        $this->getBridge()->contentUpdate($group->content_type, ['uuid' => $group->uuid]);
    }

    public function addUserToGroupMembers(string $userUuid, Groups $group): void
    {
        $user = Users::find($userUuid);
        if (! $user) {
            return;
        }

        if ($group->pendingMembers->pluck('uuid')->contains($userUuid)) {
            return;
        }

        if ($user->groups) {
            $uuids = $user->groups->pluck('uuid')->toArray();
            if (in_array($group->uuid, $uuids) || $group->uuid === null) {
                return;
            }
            $uuids[] = $group->uuid;
        } else {
            $uuids = [$group->uuid];
        }

        try {
            $this->getBridge()->contentUpdate(
                'users',
                [
                    'groups' => $uuids,
                    'uuid' => $userUuid,
                ]
            );
        } catch (Exception $e) {
            Log::error('[GroupsService::addUserToGroupMembers] failed => '.$e->getMessage());
        }

        /** @var InsideRole $role */
        $role = InsideRole::where('name', 'group-'.$group->uuid.'-member')->firstOrFail();

        Role::assignUsersToRole($role->id, [$userUuid]);
    }

    public function removeUserFromGroupAdmins(string $userUuid, Groups $group): void
    {
        $pendingMembers = $group->members->pluck('uuid')->diff([$userUuid])->all();
        $this->getBridge()->contentUpdate(
            'groups',
            [
                'uuid' => $group->uuid,
                'members' => $pendingMembers,
            ]
        );

        UserRevokedAdminEvent::dispatch($group, $userUuid);
    }

    public function removeUserFromGroupPendingMembers(string $userUuid, Groups $group): void
    {
        $pendingMembers = $group->pendingMembers->pluck('uuid')->diff([$userUuid])->all();
        $this->getBridge()->contentUpdate(
            'groups',
            [
                'uuid' => $group->uuid,
                'pending_members' => $pendingMembers,
            ]
        );
    }

    /**x
     * @throws Exception
     */
    public function removeUserFromGroupMembers(string $userUuid, Groups $group): void
    {
        $user = Users::find($userUuid);
        $uuids = $user->groups->pluck('uuid')->toArray();
        $uuids = array_diff($uuids, [$group->uuid]);
        $this->getBridge()->contentUpdate(
            'users',
            [
                'groups' => $uuids,
                'uuid' => $userUuid,
            ]
        );
        /** @var InsideRole $role */
        $role = InsideRole::where('name', 'group-'.$group->uuid.'-member')->firstOrFail();
        Role::revokeUsersToRole($role->id, [$userUuid]);
        /** @var InsideRole $role */
        $role = InsideRole::where('name', 'group-'.$group->uuid.'-admin')->firstOrFail();
        Role::revokeUsersToRole($role->id, [$userUuid]);
    }

    public function getGroupMembers(Groups $group, string $excluded = null): Collection
    {
        switch ($group->visibility) {
            case 'public':
                return Users::when(
                    $excluded != null,
                    function ($query) use ($excluded) {
                        return $query->where('uuid', '!=', $excluded);
                    }
                )->get();
            case 'private':
            case 'restricted':
                $userIds = DB::table('inside_pivots')->where('related_type', Groups::class)->where(
                    'related_uuid',
                    $group->uuid
                )->where('parent_type', Users::class)->get()->pluck('parent_uuid');

                return Users::whereIn('uuid', $excluded != null ? $userIds->except($excluded) : $userIds)->get();
            default:
                throw new InvalidArgumentException('Unknown group visibility <'.$group->visibility.'>');
        }
    }

    public function can(string $action, string $type, ?string $uuid, ?User $user): bool
    {
        if (is_null($user)) {
            /** @var ?User $user */
            $user = Auth::user();
        }
        if (is_null($user)) {
            return false;
        }

        if ($user->permission->isSuperAdmin()) {
            return true;
        }

        if (in_array($action, ['create', 'update']) && $type == 'groups') {
            $group = call_user_func(type_to_class($type).'::find', $uuid);

            return $group !== null && self::isGroupAdmin($group, $user);
        }

        if (! $this->isGroupContentType($type)) {
            return false;
        }
        $group = null;

        if ($uuid === null) {
            $request = request();
            if ($request instanceof Request && $request->has('groups')) {
                $groupUuids = $request->get('groups');
                if (! is_array($groupUuids)) {
                    $groupUuids = [$groupUuids];
                }
                $group = Groups::find(Arr::first($groupUuids));
            }
        } else {
            $model = call_user_func(type_to_class($type).'::find', $uuid);
            if (! $model->groups) {
                return false;
            }
            $group = $model->groups->first();
        }
        if ($group) {
            switch ($action) {
                case 'create':
                    return match ($type) {
                        'groups_posts' => ($group->visibility == 'public') || self::isGroupMember($group, $user),
                        default => $this->isGroupContentType($type) && self::isGroupAdmin($group, $user),
                    };
                case 'update':
                case 'delete':
                    return self::isGroupAdmin($group, $user);
            }
        } else {
            if ($action == 'create') {
                foreach (Groups::all() as $group) {
                    if (! $group->members) {
                        continue;
                    }
                    if ($group->members->pluck('uuid')->contains($user->uuid)) {
                        return true; // Admin ok => I can add
                    }
                }
            }
        }

        return false;
    }

    public function isGroupAdmin(Groups $group, mixed $user): bool
    {
        return $group->members && $group->members->pluck('uuid')->contains($user->uuid);
    }

    public function isGroupContentType(string $type): bool
    {
        return in_array($type, $this->getGroupContentTypes());
    }

    public function getGroupContentTypes(): array
    {
        return $this->groupTypes;
    }

    public function isGroupMember(Groups $group, Authenticatable $user): bool
    {
        return match ($group->visibility) {
            'public' => true,
            'private', 'restricted' => $group->reverseUsers && $group->reverseUsers->pluck('uuid')->contains($user->uuid),
            default => false,
        };
    }

    public function isGroupPendingMember(Groups $group, User $user): bool
    {
        return $group->pendingMembers->pluck('uuid')->contains($user->uuid);
    }

    public function getUserVisibleGroupsUuids(User $user): array
    {
        return Group::forUser($user)
            ->pluck('original_uuid')
            ->all();
    }

    public function groupsTypesVisibilityScope(Builder $builder): void
    {
        $user = Auth::user();

        if (! $user instanceof User) {
            return;
        }

        $visibleGroups = self::getUserVisibleGroupsUuids($user);

        $builder->whereHas('groups', fn ($query) => $query->whereIn('inside_content_groups.uuid', $visibleGroups));
    }

    public function groupsVisibilityScope(Builder $builder): void
    {
        $user = Auth::user();

        if (! $user instanceof User) {
            return;
        }

        $visibleGroups = self::getUserVisibleGroupsUuids($user);

        $builder->whereIn('inside_content_groups.uuid', $visibleGroups);
    }

    /**
     * @throws Exception
     */
    public function setPermission(string $type, ?string $uuid, string $action, bool $granted, string $role): void
    {
        if (! array_key_exists($role, $this->roles)) {
            $this->roles[$role] = DB::table('inside_roles')->where('name', $role)->first();
        }
        if (! isset($this->roles[$role])) {
            return;
        }
        $role = $this->roles[$role];
        $data = [
            'authorizable_type' => type_to_class($type),
            'authorizable_uuid' => $uuid,
            'action' => $action,
            'invert' => ! $granted,
            'children' => true,
        ];

        try {
            $permissionSchemaID = DB::table('inside_permissions_schema')->insertGetId($data);
        } catch (QueryException) {
            $permissionSchemaID = DB::table('inside_permissions_schema')->where($data)->first()->id ?? null;
        }

        if (! is_int($permissionSchemaID)) {
            return;
        }

        try {
            DB::table('inside_roles_permissions_schema')->insert(
                [
                    'role_id' => $role->id,
                    'permission_schema_id' => $permissionSchemaID,
                    'is_content_specific' => true,
                ]
            );
        } catch (QueryException $exception) {
            // Do nothing
        }

        if ($data['invert']) {
            Permission::deletePermission($role->id, $data);
        } else {
            Permission::createPermission($role->id, $data);
        }
    }

    /**
     * @param Groups $group
     * @return void
     */
    public function createNotificationsForNewGroup(Groups $group): void
    {
        $users = Users::all();
        $this->createIfNecessaryNotificationType(
            $users,
            'web',
            $group,
            GroupsDocumentsCreatedEvent::class,
            GroupsDocuments::class,
            'document',
            true
        );
        $this->createIfNecessaryNotificationType(
            $users,
            'email',
            $group,
            GroupsDocumentsCreatedEvent::class,
            GroupsDocuments::class,
            'document',
            true
        );

        // Un commentaire est publié
        $this->createIfNecessaryNotificationType(
            $users,
            'web',
            $group,
            GroupsPostsCreatedEvent::class,
            GroupsPosts::class,
            'post',
            true
        );
        $this->createIfNecessaryNotificationType(
            $users,
            'email',
            $group,
            GroupsPostsCreatedEvent::class,
            GroupsPosts::class,
            'post',
            true
        );

        // Un utilisateur est mentionné dans un groupe
        $this->createIfNecessaryNotificationType(
            $users,
            'web',
            $group,
            UserMentionedInCommentEvent::class,
            GroupsPosts::class,
            'mention',
            false
        );
        $this->createIfNecessaryNotificationType(
            $users,
            'email',
            $group,
            UserMentionedInCommentEvent::class,
            GroupsPosts::class,
            'mention',
            false
        );

        // Un commentaire est répondu
        $this->createIfNecessaryNotificationType(
            $users,
            'web',
            $group,
            GroupsPostsAnsweredEvent::class,
            GroupsPosts::class,
            'post.answer',
            false
        );
        $this->createIfNecessaryNotificationType(
            $users,
            'email',
            $group,
            GroupsPostsAnsweredEvent::class,
            GroupsPosts::class,
            'post.answer',
            false
        );

        // Reports
        // Un utilisateur a reporté un post
        $notificationType = $this->prepareNotificationTypes(
            [
                'via' => 'web',
                'default' => false,
                'event' => CustomNotificationEvent::class,
                'model' => GroupsPosts::class,
                'action' => 'report',
                'type' => 'global',
                'condition' => 'groups:'.$group->uuid,
                'multiple' => true,
                'language' => true,
                'profile' => false,
                'role' => true,
            ],
            [
                'title' => 'notifications.group.report.groups_posts.title',
                'description' => 'notifications.group.report.groups_posts.description',
                'icon' => 'group',
                'text' => 'notifications.group.report.groups_posts.text',
                'fields' => ['groups', 'authors'],
                'url' => 'ROOT_RELATION:groups',
            ]
        );
        NotificationTypeHelper::subscribeRoles(
            $notificationType,
            ['super_administrator', 'group-'.$group->uuid.'-admin'],
            true
        );
    }

    /**
     * @param Collection $users
     * @param string $via
     * @param Groups $group
     * @param string $event
     * @param string $model
     * @param string $langKey
     * @param bool $role
     * @return void
     */
    public function createIfNecessaryNotificationType(
        Collection $users,
        string $via,
        Groups $group,
        string $event,
        string $model,
        string $langKey,
        bool $role
    ): void {
        $data = $this->buildNotificationTypeData(
            $via,
            $group,
            $event,
            $model,
            $langKey,
            $role
        );
        if (! NotificationType::where(array_except($data, 'data'))->exists()) {
            /** @var NotificationType $notificationType */
            $notificationType = NotificationType::create($data);
            $notPublic = $group->visibility != 'public';
            $roleName = $notPublic ? 'group-'.$group->uuid.'-member' : 'authenticated';
            if ($role) {
                NotificationTypeHelper::subscribeRoles(
                    $notificationType,
                    $roleName,
                    !$notPublic
                );
            } elseif ($notPublic) {
                foreach ($users as $user) {
                    if ($user && ! $user->groups->contains($group->uuid)) {
                        continue;
                    }

                    NotificationSubscriber::firstOrCreate(
                        [
                            'user_uuid' => $user->uuid,
                            'notification_type_id' => $notificationType->id,
                        ]
                    );
                }
            }
        }
    }

    /**
     * @param string $via
     * @param Groups $group
     * @param string $event
     * @param string $model
     * @param string $langKey
     * @param bool $role
     * @return array
     */
    protected function buildNotificationTypeData(
        string $via,
        Groups $group,
        string $event,
        string $model,
        string $langKey,
        bool $role
    ): array {
        $data = [
            'title' => 'notifications.group.'.$via.'.'.$langKey.'.title',
            'description' => 'notifications.group.'.$via.'.'.$langKey.'.description',
            'icon' => 'group',
            'fields' => ['title', ['groups' => ['title', 'slug']]],
            'url' => 'RELATION:groups',
        ];
        if ($via == 'web') {
            $data['text'] = 'notifications.group.'.$via.'.'.$langKey.'.text';
        } elseif ($via == 'email') {
            $data['mail'] = [
                'subject' => 'notifications.group.'.$via.'.'.$langKey.'.subject',
                'text' => 'notifications.group.'.$via.'.'.$langKey.'.content',
                'buttonText' => 'notifications.group.'.$via.'.'.$langKey.'.buttonText',
            ];
        }

        return [
            'via' => $via,
            'default' => false,
            'event' => $event,
            'model' => $model,
            'action' => 'custom',
            'type' => 'group',
            'condition' => 'groups:'.$group->uuid,
            'multiple' => true,
            'language' => true,
            'profile' => false,
            'role' => $role,
            'data' => $data,
        ];
    }

    /**
     * @param array $info
     * @param array $data
     * @return mixed
     */
    protected function prepareNotificationTypes(array $info, array $data)
    {
        /** @var ?NotificationType $notificationType */
        $notificationType = NotificationType::where($info)->first();
        if ($notificationType) {
            $notificationType->data = $data;
            $notificationType->save();
        } else {
            $info['data'] = $data;
            $notificationType = NotificationType::create($info);
        }

        return $notificationType;
    }

    /**
     * @param string $type
     * @return void
     */
    public function addGroupContentType(string $type): void
    {
        if (! in_array($type, $this->getGroupContentTypes())) {
            if (! Schema::hasContentType($type)) {
                throw ModelSchemaNotFoundException::named($type);
            }
            $this->groupTypes[] = $type;
        }
    }

    /**
     * @param Group $group
     * @return void
     */
    public function updateLasGroupActivityActivity(Group $group): void
    {
        $user = Auth::user();
        if ($user === null) {
            return;
        }
        if (null === ($activity = GroupActivity::where('group_id', $group->id)->where('user_uuid', $user->uuid)->first())) {
            GroupActivity::create(
                [
                    'group_id' => $group->id,
                    'user_uuid' => $user->uuid,
                    'last_activity_at' => now(),
                ]
            );

            return;
        }
        $activity->update(
            [
                'last_activity_at' => now(),
            ]
        );
    }

    /**
     * @param GroupPost $message
     * @return bool
     */
    public function isNewMessage(GroupPost $message): bool
    {
        $user = Auth::user();
        if ($user === null) {
            return false;
        }
        $activity = GroupActivity::where('group_id', $message->group_id)->where('user_uuid', $user->uuid)->first();

        if ($activity === null) {
            return true;
        }

        return $activity->last_activity_at->isBefore($message->created_at);
    }
}
