<?php

namespace Inside\Groups\Repositories;

use Exception;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Inside\Authentication\Models\User;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Contents\ImageStyles;
use Inside\Database\Eloquent\Builder;
use Inside\Support\Str;
use Symfony\Component\HttpFoundation\File\MimeType\MimeTypeGuesser;

/**
 * @package Inside\Groups\Repositories
 *
 * @OA\Schema(schema="GroupPostAuthorResource",type="object", description="Un auteur",
 *     @OA\Property(property="uuid", title="uuid", type="string", description="Uuid",
 *                                   example="b33e94fe-4496-44d3-bd16-314872d07a74"),
 *     @OA\Property(property="firstname", title="firstname", type="string", description="Prénom", example="Léo"),
 *     @OA\Property(property="lastname", title="lastname", type="string", description="Nom", example="Brossault"),
 *     @OA\Property(property="image", title="image", type="object|null", description="L'image",
 *                                    ref="#/components/schemas/ImageResource"))
 * @OA\Schema(schema="Reaction")
 *
 * @OA\Schema(schema="FileResource", type="object", description="Un champ fichier",
 *     @OA\Property(property="mimetype", title="mimetype", type="string", description="Type mime de fichier"),
 *     @OA\Property(property="path", title="path", type="string", description="Le chemin relatif du fichier"),
 *     @OA\Property(property="size", title="size", type="integer", description="La taille en octet du fichier"),
 *     @OA\Property(property="basename", title="basename", type="string", description="La nom du fichier"),
 *     @OA\Property(property="width", title="width", type="string|null", description="La largeur de l'image si le
 *                                    fichier est une image"),
 *     @OA\Property(property="height", title="height", type="string|null", description="La hauteur de l'image si le
 *                                     fichier est une image"), example={
"mimetype": "image/png",
"path": "files/comment-b6922941-2603-481b-b48a-f24052947545-gprpe9gph/2019-11/mon-image.png",
"size": 2901820,
"basename": "mon-image.png",
"width": 1414,
"height": 2000
}
 * )
 *
 *
 * @OA\Schema(schema="ImageResource",type="object", description="Un champ fichier",
 *     @OA\AdditionalProperties(type="string", description="Type mime de fichier"),
 *     example={"main":
 *     "files/b33e94fe-4496-44d3-bd16-314872d07a74/users/2019-06/tpvI7dgI7o9N25ei.jpg","thumbnail":"files/c55e94fe-4496-44d3-bd16-314872d07a74/users/2019-06/tpvI7dgI7o9N25ei.jpg"}
 * )
 *
 */
class Repository
{
    protected function getAuthor(User $user): array
    {
        $anonymousInformation = (object)config(
            'contents.anonymous_users.fields',
            [
                'firstname' => 'Utilisateur',
                'lastname'  => 'Anonyme',
            ]
        );

        return [
            'uuid' => $user->status ? $user->uuid : '',
            'firstname' => $user->status && $user->information !== null
                ? $user->information->firstname
                : trans(
                    $anonymousInformation->firstname
                ),
            'lastname' => $user->status && $user->information !== null
                ? $user->information->lastname
                : trans(
                    $anonymousInformation->lastname
                ),
            'image' => $user->status && $user->information !== null
                ? $this->prepareImageResult($user->information, 'image')
                : null,
        ];
    }

    protected function prepareImageResult(Model $content, string $fieldName): ?array
    {
        $path = $content->{$fieldName};
        if ($path === null || !Storage::disk('local')->exists($path)) {
            return null;
        }
        $protectedPath = protected_file_url($content, $fieldName);
        $value = [
            'main' => $protectedPath,
        ];
        $options = Schema::getFieldOptions(class_to_type($content), $fieldName);
        if (isset($options['image_styles']) && !empty($options['image_styles'])) {
            $styles = ImageStyles::find($options['image_styles']);
            foreach ($styles as $style) {
                $value[Str::slug($style->title)] = protected_file_url($content, $fieldName, true, $style->title);
            }
        }

        return $value;
    }

    protected function prepareFileResult(Model $content, string $fieldName): ?array
    {
        $path = $content->{$fieldName};
        if ($path === null || !Storage::disk('local')->exists($path)) {
            return null;
        }
        $realPath = Storage::disk('local')->path($path);
        $value    = [
            'mimetype' => '',
            'path' => protected_file_url($content, $fieldName),
            'size' => '',
            'basename' => '',
        ];

        if (is_string($realPath) && !empty($realPath) && File::exists($realPath)) {
            $value['basename'] = basename($realPath);
            $guesser = MimeTypeGuesser::getInstance();
            $value['mimetype'] = $guesser->guess($realPath);
            $value['size'] = filesize($realPath);
            $width = $height = null;

            try {
                [$width, $height] = getimagesize($realPath);
            } catch (Exception $exception) {
            }

            if ($width !== null && $height !== null) {
                $value['width'] = $width;
                $value['height'] = $height;
            }
        }

        return $value;
    }

    protected function getTermsFromQueryString(?string $query): Collection
    {
        $terms = collect();
        if (isset($query)) {
            $terms = Str::of($query)->replace([',', ';'], ' ')->explode(' ');
        }

        return $terms;
    }

    protected function addSearchTermsToQuery(Builder $query, string $column, Collection $terms): Builder
    {
        $query->where($column, 'like', '%' . $terms->implode(' ') . '%');
        foreach ($terms as $term) {
            $query->orWhere($column, 'like', '%' . $term . '%');
        }

        return $query;
    }
}
