<?php

namespace Inside\Groups\Repositories;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Inside\Authentication\Models\User;
use Inside\Groups\Facades\GroupsHelper;
use Inside\Groups\Models\Group;
use Inside\Groups\Models\GroupActivity;
use Inside\Groups\Models\GroupPost;
use OpenApi\Annotations as OA;

/**
 * @package Inside\Groups\Repositories
 */
class GroupRepository extends Repository
{
    /**
     * @OA\Schema(schema="GroupResource",
     * @OA\Property(property="uuid", title="uuid", type="string", description="Uuid"),
     * @OA\Property(property="name", title="name", type="string", description="Nom du groupe", example="Mon premier
     *                               groupe"),
     * @OA\Property(property="type", title="type", type="string", enum={"public","restricted","private"},
     *                                   description="Type de groupe", default="public"),
     * @OA\Property(property="slug", title="slug", type="string", description="Slug unique pour la génération de
     *                               l'url", example="mon-groupe"),
     * @OA\Property(property="last_activity", title="last_activity", type="integer", description="Dernière activité
     *                                            dans le groupe ( en timestamp )", example=1621584775),
     * @OA\Property(property="posts_count", title="posts_count", type="integer", description="Nombre de messages total
     *                                      dans le groupe", example=10),
     * @OA\Property(property="image", title="image", type="object|null", optionnal=true, description="L'image
     *                                d'illustration du groupe", ref="#components/schemas/FileResource"),
     * @OA\Property(property="new_posts_count", title="new_posts_count", type="integer", description="Nombre de
     *                                          messages depuis la dernière visite", example=2),
     * @OA\Property(property="can_update", title="can_update", type="boolean", description="L'utilisateur courant
     *                                         peut-il modifier ce groupe?", example=true),
     * @OA\Property(property="can_delete", title="can_delete", type="boolean", description="L'utilisateur courant
     *                                         peut-il supprimer ce groupe?", example=true),
     * @OA\Property(property="posts_found", title="posts_found", type="integer", optionnal=true, description="Nombre de
     *                                            message(s) dans le groupe si une recherche est effectuée avec les
     *                                            termes demandées", example=3),
     * @OA\Property(property="is_pending", title="is_pending", type="boolean", optionnal=true, description="Indique si
     *                                     l'utilisateur courant est en attente d'approbation pour rejoindre le groupe
     *                                     ( uniquement si le groupe est de type restreint )", example=false),
     * @OA\Property(property="admins", title="admins", type="array", description="Les admins du groupe",
     *                                 @OA\Items(ref="#/components/schemas/GroupPostAuthorResource")),
     * @OA\Property(property="members", title="members", type="array", description="Les membres du groupe (non
     *                                  disponible pour les groupes publiques )",
     * @OA\Items(ref="#/components/schemas/GroupPostAuthorResource"))
     * )
     *
     * @OA\Schema(schema="GroupList",title="Group List",type="array",description="Liste des groupes",
     *          @OA\Items(ref="#/components/schemas/GroupResource"))
     *
     *
     */
    public function list(?string $query = null, ?int $limit = null): mixed
    {
        $terms = $this->getTermsFromQueryString($query);
        /** @var User $user */
        $user = Auth::user();
        $query = Group::forUser($user)
            ->whereEnabled(true)
            ->select('groups.*')
            ->with(['admins'])
            ->withCount('nonAdmins')
            ->addSelect(
                [
                    'posts_found' => GroupPost::from('group_posts as gp')
                        ->whereColumn('gp.group_id', 'groups.id')
                        ->selectRaw('count(*)')
                        ->where(fn ($query) => $this->addSearchTermsToQuery($query, 'gp.body', $terms))
                        ->take(1),
                    'never_visits' => GroupActivity::from('group_activities as gans')
                        ->selectRaw('count(*)')
                        ->whereColumn('gans.group_id', 'groups.id')
                        ->where('gans.user_uuid', $user->uuid)
                        ->take(1),
                    'new_posts_count' => GroupPost::from('group_posts as igp')
                        ->selectRaw('count(*)')
                        ->join('group_activities as ga', 'igp.group_id', 'ga.group_id')
                        ->where('ga.user_uuid', $user->uuid)
                        ->whereColumn('igp.group_id', 'groups.id')
                        ->whereColumn('igp.created_at', '>', 'ga.last_activity_at')
                        ->take(1),
                ]
            );

        if ($terms->isNotEmpty()) {
            $query->having('posts_found', '>', '0')->orHaving('groups.name', 'like', '%'.$terms->implode(' ').'%');
        }

        if ($limit !== null) {
            $query->take($limit);
        }

        return tap($query->latest('last_activity')->orderBy('groups.name')->get())->transform(
            function ($group) use ($terms) {
                return $this->presentGroup($group, $terms);
            }
        )->filter()->values();
    }

    protected function presentGroup(Group $group, Collection $terms): ?array
    {
        $presentedGroup = [];
        if ($terms->isNotEmpty()) {
            $presentedGroup = ['posts_found' => $group->posts_found ?? [],];
        }
        /** @var User $user */
        $user = Auth::user();
        $isAdmin = $user->isSuperAdmin() || $group->admins->contains($user);
        $presentedGroup['posts_count'] = $group->posts_count;
        $presentedGroup['last_activity'] = $group->last_activity->timestamp ?? null;
        $presentedGroup['new_posts_count'] = $group->never_visits === 0 ? $group->posts_count : $group->new_posts_count;
        $presentedGroup['in_group'] = $isAdmin || $group->members->contains($user);

        if ($group->type !== 'public') {
            $presentedGroup['non_admins_count'] = $group->non_admins_count;
            if ($group->type === 'restricted') {
                $presentedGroup['is_pending'] = $group->pendingUsers->contains($user);
                if (! $user->isSuperAdmin() && ! $group->members->contains($user)) {
                    $presentedGroup['posts_count'] = 0;
                    $presentedGroup['last_activity'] = null;
                    $presentedGroup['new_posts_count'] = 0;
                }
            }
        }

        return array_merge(
            [
                'uuid' => $group->original_uuid,
                'name' => $group->name,
                'type' => $group->type,
                'slug' => $group->slug,
                'image' => $this->prepareFileResult($group->original, 'image'),
                'can_update' => $isAdmin,
                'can_delete' => $isAdmin,
                'admins' => tap($group->admins)->transform(
                    function ($admin) {
                        return $this->getAuthor($admin);
                    }
                ),
            ],
            $presentedGroup
        );
    }

    /**
     * @OA\Schema(schema="SimpleGroupResource",
     * @OA\Property(property="name", title="name", type="string", description="Nom du groupe", example="Mon premier
     *                               groupe"),
     * @OA\Property(property="type", title="type", type="string", enum={"public","restricted","private"},
     * @OA\Property(property="slug", title="slug", type="string", description="Slug unique pour la génération de
     *                               l'url", example="mon-groupe"), description="Type de groupe", default="public"),
     * @OA\Property(property="last_activity", title="last_activity", type="integer", description="Dernière activité
     *                                            dans le groupe ( en timestamp )", example=1621584775),
     * @OA\Property(property="new_posts_count", title="new_posts_count", type="integer", description="Nombre de
     *                                          messages depuis la dernière visite", example=2),
     * @OA\Items(ref="#/components/schemas/GroupPostAuthorResource"))
     * )
     *
     * @OA\Schema(schema="SimpleGroupList",title="Simple Group List",type="array",description="Liste des groupes (
     *                                                   version simplifiée )",
     * @OA\Items(ref="#/components/schemas/SimpleGroupResource"))
     *
     */
    public function enumerate(?int $limit = null): mixed
    {
        /** @var User $user */
        $user = Auth::user();
        $query = Group::forUser($user)
            ->whereEnabled(true)
            ->select('groups.*')
            ->with(['admins', 'members'])
            ->addSelect(
                [
                    'never_visits' => GroupActivity::from('group_activities as gans')
                        ->selectRaw('count(*)')
                        ->whereColumn('gans.group_id', 'groups.id')
                        ->where('gans.user_uuid', $user->uuid)
                        ->take(1),
                    'posts_count' => GroupPost::from('group_posts as gp')
                        ->selectRaw('count(*)')->whereColumn('gp.group_id', 'groups.id')
                        ->take(1),
                    'new_posts_count' => GroupPost::from('group_posts as igp')
                        ->selectRaw('count(*)')
                        ->join('group_activities as ga', 'igp.group_id', 'ga.group_id')
                        ->where('ga.user_uuid', $user->uuid)
                        ->whereColumn('igp.group_id', 'groups.id')
                        ->whereColumn('igp.created_at', '>', 'ga.last_activity_at')
                        ->take(1),
                ]
            );

        if ($limit !== null) {
            $query->limit($limit);
        }

        return tap($query->latest('last_activity')->orderBy('name')->get())->transform(
            function ($group) {
                return [
                    'name' => $group->name,
                    'slug' => $group->slug,
                    'type' => $group->type,
                    'last_activity' => $group->last_activity->timestamp,
                    'new_posts_count' => $group->never_visits === 0 ? $group->posts_count : $group->new_posts_count,
                ];
            }
        );
    }
}
