<?php

namespace Inside\Groups\Repositories;

use Illuminate\Container\Container;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Pagination\Paginator;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\HigherOrderTapProxy;
use Inside\Authentication\Models\User;
use Inside\Content\Contracts\Transformer;
use Inside\Content\Facades\ContentHelper;
use Inside\Database\Eloquent\Builder;
use Inside\Groups\Facades\GroupsHelper;
use Inside\Groups\Models\Group;
use Inside\Groups\Models\GroupPost;
use Inside\Permission\Exceptions\AuthorizationException;
use Inside\Support\Str;
use OpenApi\Annotations as OA;

/**
 * @package Inside\Groups\Repositories
 */
class GroupPostRepository extends Repository
{
    protected string $highlightPrefix = '<span class="highlight">';

    protected string $highlightSuffix = '</span>';

    protected User $user;

    public function __construct(
        protected Transformer $transformer
    ) {
        /** @var User $user */
        $user = Auth::user();
        $this->user = $user;
    }

    /**
     * @OA\Schema(schema="GroupPostResource",
     * @OA\Property(property="uuid", title="uuid", type="string", description="Uuid"),
     * @OA\Property(property="body", title="body", type="string", description="Message", example="Ceci est mon
     *                                   message"),
     * @OA\Property(property="file", title="file", type="object|null", description="Chemin du fichier si non null",
     *                                   ref="#components/schemas/FileResource"),
     * @OA\Property(property="author", title="author", type="object", description="Auteur du message",
     *                                     ref="#/components/schemas/GroupPostAuthorResource"),
     * @OA\Property(property="is_new", title="is_new", type="boolean", description="Indique si ce message est
     *                                     nouveau depuis la dernière visite de ce groupe", example=false),
     * @OA\Property(property="can_delete", title="can_delete", type="boolean", description="L'utilisateur courant
     *                                         peut-il supprimer ce message?", example=true),
     * @OA\Property(property="created_at", title="created_at", type="integer", description="Date du message ( en
     *                                         timestamp )", example=1621584775),
     * @OA\Property(property="likes", title="likes", type="array", description="Les utilisateurs qui ont likés le
     *                                    message",
     *                                        @OA\Items(ref="#/components/schemas/GroupPostAuthorResource")),
     * @OA\Property(property="likes_count", title="likes_count", type="integer", description="Nombre de
     *                                            like(s) pour ce message", example=3),
     * @OA\Property(property="replies_count", title="replies_count", type="integer", description="Nombre de
     *                                            réponse(s)", example=3),
     * @OA\Property(property="replies_found", title="replies_found", type="integer", description="Nombre de
     *                                            réponse(s) comportant le ou les terme(s) recherché(s). (N'est
     *                                            disponible que si une recherche est effectuée)", example=4),
     * @OA\Property(property="replies", title="replies", type="array", description="Réponse(s) au message",
     *                                      @OA\Items(ref="#/components/schemas/GroupPostResource"))
     * )
     *
     * @OA\Schema(schema="GroupPostList",title="Group Post List",type="array",description="A group post list tree",
     * @OA\Items(ref="#/components/schemas/GroupPostResource"))
     *
     *
     * @param Group $group
     * @param string|null $query
     * @param int|null $limit
     * @param bool $withPagination
     * @return \Illuminate\Contracts\Pagination\LengthAwarePaginator|\Illuminate\Contracts\Pagination\Paginator|\Illuminate\Database\Eloquent\Collection|HigherOrderTapProxy|Builder[]|mixed
     */
    public function list(
        Group $group,
        ?string $query = null,
        ?int $limit = null,
        bool $withPagination = false
    ) {
        $user = $this->user ?? Auth::user();
        if (! $user instanceof User || ! $group->canUserAccess($user)) {
            throw new ModelNotFoundException($group);
        }

        $terms = $this->getTermsFromQueryString($query);
        $query = GroupPost::query()
            ->where('group_posts.group_id', $group->id)
            ->whereNull('group_posts.parent_id')
            ->whereHas('original', fn (Builder $query) => $query->where('status', 1))
            ->with(['replies' => fn (HasMany $query) => $query->oldest()])
            ->when(! empty($terms), fn (Builder $query) => $query->addSelect(
                [
                    'found' => GroupPost::from('group_posts as gp')
                        ->whereColumn('gp.id', 'group_posts.id')
                        ->selectRaw('count(*)')
                        ->where(fn (Builder $query) => $this->addSearchTermsToQuery($query, 'gp.body', $terms))
                        ->take(1),
                    'replies_found' => GroupPost::from('group_posts as gpr')
                        ->selectRaw('count(*)')
                        ->whereColumn('gpr.parent_id', 'group_posts.id')
                        ->where(fn (Builder $query) => $this->addSearchTermsToQuery($query, 'gpr.body', $terms))
                        ->take(1),
                ]
            ))
            ->latest();

        if (! empty($terms)) {
            $query->having('found', '>', 0)->orHaving('replies_found', '>', 0);
        }

        if ($limit !== null && $withPagination) {
            $currentPage = Paginator::resolveCurrentPage();
            $total = ContentHelper::getComplexQuerySize($query);
            $results = Container::getInstance()->makeWith(
                LengthAwarePaginator::class,
                [
                    'items' => $query->forPage($currentPage, $limit)->get(),
                    'total' => $total,
                    'perPage' => $limit,
                    'currentPage' => $currentPage,
                    'options' => [
                        'path' => Paginator::resolveCurrentPath(),
                        'pageName' => 'page',
                    ],
                ]
            );
        } else {
            if ($limit !== null) {
                $query->take($limit);
            }
            $results = $query->get();
        }

        $results = tap($results)->transform(
            function ($message) use ($terms) {
                return $this->presentMessage($message, $terms);
            }
        );

        if ($terms->isEmpty()) {
            GroupsHelper::updateLasGroupActivityActivity($group);
        }

        return $results;
    }

    /**
     * @param string $uuid
     * @param string|null $query
     * @return array
     */
    public function get(string $uuid, ?string $query): array
    {
        $user = Auth::user();

        if (Str::isUuid($uuid)) {
            $post = GroupPost::where('original_uuid', $uuid)->firstOrFail();
        } else {
            $post = GroupPost::findOrFail($uuid);
        }

        /** @var GroupPost $post */
        $post->load(
            [
                'replies' => function ($query) {
                    $query->oldest();
                },
            ]
        );

        if (! $user instanceof User || ! $post->group->canUserAccess($user)) {
            throw AuthorizationException::create('read', 'groups', $post->group->original_uuid);
        }

        return $this->presentMessage($post, $this->getTermsFromQueryString($query));
    }

    /**
     * @param GroupPost $message
     * @param Collection $terms
     * @return array
     */
    protected function presentMessage(GroupPost $message, Collection $terms): array
    {
        /** @var User $user */
        $user = $this->user ?? Auth::user();
        $presentedMessage = [];
        if ($terms->isNotEmpty()) {
            $presentedMessage = ['replies_found' => data_get($message, 'replies_found', [])];
        }

        $likes = $message->reactions->where('type', 'like')->transform(
            function ($reaction) {
                /** @var User $author */
                $author = User::find($reaction->user_uuid);
                return $this->getAuthor($author);
            }
        )->toArray();

        return array_merge(
            [
                'uuid' => $message->original_uuid,
                'file' => $this->prepareFileResult($message->original, 'file'),
                'body' => $this->highlightMessage($message->body, $terms),
                'author' => $this->getAuthor($message->author),
                'created_at' => $message->created_at->timestamp,
                'likes' => $likes,
                'likes_count' => count($likes),
                'replies_count' => $message->replies_count,
                'can_delete' => $user->isSuperAdmin() || $message->author_uuid === $user->uuid || $message->group->admins->contains($user),
                'is_new' => GroupsHelper::isNewMessage($message),
                'replies' => tap(
                    $message->replies,
                    function ($replies) use ($terms) {
                        return $replies->transform(
                            function ($message) use ($terms) {
                                return $this->presentMessage($message, $terms);
                            }
                        );
                    }
                ),
            ],
            $presentedMessage
        );
    }

    /**
     * @param string $message
     * @param Collection $terms
     * @return null|string
     */
    protected function highlightMessage(string $message, Collection $terms): ?string
    {
        $message = iclean($message);
        if (empty($terms)) {
            return $message;
        }

        return preg_replace(
            $terms->map(
                function ($term) {
                    return '#'.preg_quote($term).'#i';
                }
            )->toArray(),
            $terms->map(
                function () {
                    return $this->highlightPrefix.'\\0'.$this->highlightSuffix;
                }
            )->toArray(),
            $message
        );
    }
}
