<?php

namespace Inside\Groups\Listeners;

use Illuminate\Events\Dispatcher;
use Illuminate\Support\Facades\Log;
use Inside\Content\Events\ContentDeletedEvent;
use Inside\Content\Events\ContentFullyInsertedEvent;
use Inside\Content\Events\ContentFullyUpdatedEvent;
use Inside\Groups\Models\Group;
use Inside\Groups\Models\GroupPost;
use Throwable;

class SynchronizeGroupPost
{
    public function handleGroupPostCreation(ContentFullyInsertedEvent $event): void
    {
        if (class_to_type($event->model) !== 'groups_posts') {
            return;
        }
        $postContent = $event->model;

        if ($postContent->groups === null || $postContent->groups->isEmpty()) {
            return;
        }

        $group = Group::where('original_uuid', $postContent->groups->first()->uuid)->first();
        if ($group === null) {
            return;
        }

        $parent = GroupPost::where('original_uuid', $postContent->pid)->first();

        try {
            GroupPost::create(
                [
                    'group_id'      => $group->id,
                    'parent_id'     => $parent->id,
                    'body'          => $postContent->body,
                    'file'          => $postContent->file,
                    'original_uuid' => $postContent->uuid,
                    'author_uuid'   => $postContent->author,
                ]
            );
        } catch (Throwable $e) {
            Log::warning('[group_posts] Failed to sync group post ( creation ) => ' . $e->getMessage());
        }
    }

    public function handleGroupPostModification(ContentFullyUpdatedEvent $event): void
    {
        if (class_to_type($event->model) !== 'groups_posts') {
            return;
        }
        $postContent = $event->model;
        try {
            GroupPost::where('original_uuid', $postContent->uuid)
                ->update(
                    [
                        'body' => $postContent->body,
                        'file' => $postContent->file,
                    ]
                );
        } catch (Throwable $e) {
            Log::error('[group_posts] Failed to sync group post ( modification ) => ' . $e->getMessage());
        }
    }

    public function handleGroupPostDeletion(ContentDeletedEvent $event): void
    {
        if (class_to_type($event->model) !== 'groups_posts') {
            return;
        }
        try {
            $postContent = GroupPost::find($event->model->uuid);
            if ($postContent !== null) {
                $postContent->delete();
            }
        } catch (Throwable $e) {
            Log::error('[group_posts] Failed to sync group post ( deletion ) => ' . $e->getMessage());
        }
    }

    public function subscribe(Dispatcher $events): array
    {
        return [
            ContentFullyInsertedEvent::class => 'handleGroupPostCreation',
            ContentFullyUpdatedEvent::class => 'handleGroupPostModification',
            ContentDeletedEvent::class => 'handleGroupPostDeletion',
        ];
    }
}
