<?php

namespace Inside\Groups\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Inside\Authentication\Models\User;
use Inside\Content\Events\ContentInsertedEvent;
use Inside\Content\Models\Contents\Groups;
use Inside\Content\Models\Contents\GroupsPosts;
use Inside\Groups\Events\GroupsPostsAnsweredEvent;
use Inside\Groups\Events\GroupsPostsCreatedEvent;
use Inside\Groups\Models\Group;
use Inside\Notify\Listeners\BaseNotificationListener;
use Inside\Notify\Models\NotificationType;

class GroupsPostsListener extends BaseNotificationListener implements ShouldQueue
{
    public function handle(ContentInsertedEvent $event): void
    {
        $post = $event->model;

        if (! $post instanceof GroupsPosts) {
            return;
        }

        $groupModel = $post->groups?->first();

        if (
            ! $groupModel instanceof Groups
            || $post->author === null
            || $post->post_type === 'system'
        ) {
            return;
        }

        $group = Group::whereOriginalUuid($groupModel->uuid)->first();

        if (! $group instanceof Group) {
            return;
        }

        if ($post->pid === null) {
            $this->groupPostCreated($post, $group);
        } else {
            $this->groupPostAnswered($post, $group);
        }
    }

    private function groupPostAnswered(GroupsPosts $post, Group $group): void
    {
        $parent = $post->parent;

        if (! $parent instanceof GroupsPosts || $post->author === $parent->author) {
            return;
        }

        /** @var User $subscriber */
        $subscriber = User::find($parent->author);

        NotificationType::query()
            ->where('model', GroupsPosts::class)
            ->where('type', 'group')
            ->where('action', 'custom')
            ->where('condition', "groups:$group->original_uuid")
            ->where('event', GroupsPostsAnsweredEvent::class)
            ->each(function (NotificationType $type) use ($post, $subscriber) {
                $this->when = $this->getInterval($type, $post);
                $this->route = $subscriber;

                if (
                    $post->author == $subscriber->uuid
                    || ! $subscriber->status
                    || ! $subscriber->hasToBeNotified($type->id)
                    || ! $this->checkUniqueness($type, $post, $subscriber)
                ) {
                    return;
                }

                $this->notify($type, $subscriber, $post, ['from' => $post->author]);
            });
    }

    private function groupPostCreated(GroupsPosts $post, Group $group): void
    {
        NotificationType::query()
            ->where('model', GroupsPosts::class)
            ->where('type', 'group')
            ->where('action', 'custom')
            ->where('event', GroupsPostsCreatedEvent::class)
            ->where('condition', "groups:$group->original_uuid")
            ->each(function (NotificationType $type) use ($group, $post) {
                $this->when = $this->getInterval($type, $post);

                if ($type->role && $type->via === 'web') {
                    if ($type->roles->first()) {
                        $this->route = $type->roles->first();
                        $this->notify($type, null, $post, ['from' => $post->author]);
                    }

                    return;
                }

                $group
                    ->members
                    ->reject(fn (User $member) => $member->uuid === $post->author
                        || ! $member->status
                        || ! $member->hasToBeNotified($type->id)
                        || ! $this->checkUniqueness($type, $post, $member))
                    ->each(function (User $member) use ($post, $type) {
                        $this->route = $member;
                        $this->notify($type, $member, $post, ['from' => $post->author]);
                    });
            });
    }
}
