<?php

namespace Inside\Groups\Listeners;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Inside\Content\Events\ContentInsertedEvent;
use Inside\Content\Models\Contents\Groups;
use Inside\Groups\Facades\GroupsHelper;
use Inside\Notify\Models\NotificationSubscriber;
use Inside\Notify\Models\NotificationType;
use Inside\Permission\Models\Role;
use Inside\User\Models\User;

class GroupsListener
{
    /**
     * @var array<int, string>
     */
    protected array $roles = [];

    public function handle(ContentInsertedEvent $event): void
    {
        if (isset($event->model) && $event->model instanceof Groups) {
            $this->createGroupRoles($event->model);
            GroupsHelper::createNotificationsForNewGroup($event->model);
            $this->handleNewGroup($event->model);
        }
    }

    /**
     * @param Groups $group
     * @return void
     */
    protected function createGroupRoles(Groups $group): void
    {
        Role::firstOrCreate(
            [
                'name' => 'group-'.$group->uuid.'-member',
                'type' => 'group',
            ]
        );

        Role::firstOrCreate(
            [
                'name' => 'group-'.$group->uuid.'-admin',
                'type' => 'group',
            ]
        );
    }

    /**
     * @param Groups $group
     * @return void
     */
    protected function handleNewGroup(Groups $group): void
    {
        /** @var User $me */
        $me = Auth::user();
        GroupsHelper::addUserToGroupMembers($me->uuid ?? $group->author_id, $group);
        /** @var Collection<NotificationType> $notificationTypes */
        $notificationTypes = NotificationType::where('condition', 'groups:'.$group->uuid)->where(
            'action',
            'custom'
        )->get();

        if ($group->visibility != 'public') {
            foreach ($notificationTypes as $notificationType) {
                NotificationSubscriber::firstOrCreate(
                    [
                        'user_uuid' => $me->uuid ?? $group->author,
                        'notification_type_id' => $notificationType->id,
                    ]
                );
            }
        }
    }
}
