<?php

namespace Inside\Groups\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Collection;
use Inside\Authentication\Models\User;
use Inside\Content\Events\ContentInsertedEvent;
use Inside\Content\Models\Contents\GroupsDocuments;
use Inside\Groups\Events\GroupsDocumentsCreatedEvent;
use Inside\Groups\Facades\GroupsHelper;
use Inside\Notify\Listeners\BaseNotificationListener;
use Inside\Notify\Models\NotificationType;
use Illuminate\Support\Facades\Lang;

class GroupsDocumentsListener extends BaseNotificationListener implements ShouldQueue
{
    /**
     * @param ContentInsertedEvent $event
     * @return void
     */
    public function handle(ContentInsertedEvent $event)
    {
        if (isset($event->model->content_type) && ($event->model->content_type == 'groups_documents')) {

            // if the document is a message attachment, we don't want to notify
            if ($event->model->groupsFolders?->first()?->title === Lang::get('folders.attachements.title')) {
                return;
            }

            $groups = $event->model->groups;

            if ($groups === null || $groups->isEmpty()) {
                return;
            }

            if ($event->model->author === null) {
                return;
            }

            /** @var Collection<NotificationType> $types */
            $types = NotificationType::where('model', GroupsDocuments::class)->where('type', 'group')->where(
                'action',
                'custom'
            )->where('condition', 'groups:' . $groups->first()->uuid)->where(
                'event',
                GroupsDocumentsCreatedEvent::class
            )->get();

            $members = GroupsHelper::getGroupMembers($groups->first(), $event->model->author);
            $subscribers = User::whereIn('uuid', $members->pluck('uuid'))->get();

            foreach ($types as $type) {
                if ($type->role && $type->via === 'web') {
                    if ($type->roles->first()) {
                        $this->route = $type->roles->first();
                        $this->notify($type, null, $event->model, ['from' => $event->model->author]);
                    }
                    continue;
                }

                $this->when = $this->getInterval($type, $event->model);
                foreach ($subscribers as $subscriber) {
                    $this->route = $subscriber;
                    if ($event->model->author == $subscriber->uuid) {
                        continue;
                    }

                    if (
                        !$subscriber->status
                        || !$subscriber->hasToBeNotified($type->id)
                        || !$this->checkUniqueness($type, $event->model, $subscriber)
                    ) {
                        continue;
                    }

                    $this->notify($type, $subscriber, $event->model, ['from' => $event->model->author]);
                }
            }
        }
    }
}
