<?php

namespace Inside\Groups\Http\Middlewares;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\Groups;
use Inside\Groups\Facades\GroupsHelper;
use Inside\Permission\Exceptions\AuthorizationException;
use Symfony\Component\Serializer\Exception\InvalidArgumentException;

class GroupContentsPermissionMiddleware
{
    public function handle(Request $request, Closure $next): mixed
    {
        $path = $request->path();
        $method = $request->getMethod();
        $implode = implode('|', GroupsHelper::getGroupContentTypes());
        $pattern = '#^api/v1/content/('.$implode.')/?([\da-f]{8}-[\da-f]{4}-[\da-f]{4}-[\da-f]{4}-[\da-f]{12})?$#';
        $matches = [];
        $group = null;

        if (
            preg_match($pattern, $path, $matches)
            && in_array($method, ['DELETE', 'PUT', 'POST'])
        ) {
            /** @var ?User $me */
            $me = Auth::user();
            if (! $me) {
                return $next($request); // Auth will fail
            }

            /** @var User $user */
            $user = User::find($me->uuid);
            if ($user->hasRole('super_administrator')) {
                return $next($request);
            }

            $type = $matches[1];

            if ($method != 'DELETE') {
                $groups = $request->get('groups', []);
                if (empty($groups)) {
                    throw new InvalidArgumentException('missing groups');
                }
                $group = Groups::find(is_array($groups) ? $groups[0] : $groups);
                if (! $group) {
                    throw AuthorizationException::create($method, $type);
                }
            }

            switch ($method) {
                case 'POST':
                    // Creation
                    // Rappel des permissions
                    // un membre peut poster dans un groupe visible, ou un groupe restreint/privé s'il en est membre
                    // seul un admin peut poster un dossier/document de ce groupe
                    switch ($type) {
                        case 'groups_posts':
                            if (! GroupsHelper::isGroupMember($group, $me)) {
                                throw AuthorizationException::create('create', $type);
                            }
                            break;
                        default:
                            if (! GroupsHelper::isGroupMember($group, $me) && ! $group->members->contains($me)) {
                                throw AuthorizationException::create('create', $type);
                            }
                            break;
                    }
                    break;
                case 'PUT':
                case 'DELETE':
                    // Edition / Deletion
                    // Rappel des permissions
                    // un membre ne peut éditer/effacer que ses publications alors qu'un admin peut tout éditer/effacer
                    // seul un admin peut éditer/effacer un dossier/document de ce groupe
                    $uuid = $matches[2];
                    $content = call_user_func(type_to_class($type).'::find', $uuid);
                    if ($content === null) {
                        throw new InvalidArgumentException();
                    }
                    if ($group === null && $content->groups) {
                        $group = $content->groups->first();
                    }
                    if ($group === null) {
                        throw new InvalidArgumentException(); // didn't find group
                    }
                    switch ($type) {
                        case 'groups_posts':
                            if (! GroupsHelper::isGroupAdmin($group, $me) && ($content->author != $me->uuid)) {
                                throw AuthorizationException::create($method == 'PUT' ? 'update' : 'delete', $type);
                            }
                            break;
                        default:
                            if (! GroupsHelper::isGroupMember($group, $me) && ! GroupsHelper::isGroupAdmin($group, $me)) {
                                throw AuthorizationException::create($method == 'PUT' ? 'update' : 'delete', $type);
                            }
                            break;
                    }
                    break;
            }
        }

        return $next($request);
    }
}
