<?php

namespace Inside\Groups\Services;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\Groups;
use Inside\Groups\Events\UserGrantedAdminEvent;
use Inside\Groups\Events\UserJoinedAGroupEvent;
use Inside\Groups\Events\UserLeftAGroupEvent;
use Inside\Groups\Events\UserRevokedAdminEvent;
use Inside\Groups\Facades\GroupsHelper;
use Inside\Groups\Jobs\AddNewGroupMembers;
use Inside\Groups\Jobs\DeleteUsersFromGroup;
use Inside\Groups\Models\Group;
use Inside\Notify\Models\NotificationType;
use Inside\Permission\Facades\Role as RoleHelper;
use Inside\Permission\Models\Role;

class GroupsMembersService
{
    public function removeUndesiredGroupMembers(Groups $originalGroup): void
    {
        $group = Group::whereOriginalUuid($originalGroup->uuid)->first();

        if (! $group instanceof Group) {
            Log::error('[inside-groups] GroupsMembersService::removeUndesiredGroupMembers() : group not found in table "groups"', ['original_uuid' => $originalGroup->uuid]);
            return;
        }

        $roleIds = $group->roles->pluck('id');

        if ($roleIds->isEmpty()) {
            return;
        }

        $undesiredMembers = $group
            ->users()
            ->whereDoesntHave('roles', fn ($query) => $query->whereIn('role_id', $roleIds))
            ->withPivot('type')
            ->get()
            ->reject(fn (User $user) => $user->isSuperAdmin() || $group->author_uuid === $user->uuid);

        $undesiredMembers->each(function (User $user) use ($group, $originalGroup) {
            GroupsHelper::removeUserFromGroupMembers($user->uuid, $group->original);

            match ($user->pivot->type) {
                'pending' => GroupsHelper::removeUserFromGroupPendingMembers($user->uuid, $group->original),
                'admin' => GroupsHelper::removeUserFromGroupAdmins($user->uuid, $group->original),
                default => null,
            };
            // Re-save to trigger events now that we have our members
            GroupsHelper::touch($originalGroup);
        });
    }

    public function unsubscribeUndesiredSubscribers(Groups $originalGroup): void
    {
        $group = Group::whereOriginalUuid($originalGroup->uuid)->first();

        if (! $group instanceof Group) {
            Log::error('[inside-groups] GroupsMembersService::unsubscribeNonMembersFromNotifications() : group not found in table "groups"', ['original_uuid' => $originalGroup->uuid]);
            return;
        }

        $roles = match (true) {
            $group->type !== 'public' => Role::where('name', "group-$group->original_uuid-member")->pluck('id'),
            $group->roles->isEmpty() => Role::where('name', 'authenticated')->pluck('id'),
            default => $group->roles->pluck('id'),
        };

        $members = $group->members->pluck('uuid');

        NotificationType::query()
            ->where('condition', "groups:$group->original_uuid")
            ->where('action', 'custom')
            ->each(function (NotificationType $notificationType) use ($group, $members, $roles) {
                $undesired = $notificationType
                    ->subscribers()
                    ->whereDoesntHave('roles', fn (Builder $query) => $query->whereIn('role_id', $roles))
                    ->get()
                    ->reject(fn (User $user) => $user->isSuperAdmin() || $group->author_uuid === $user->uuid)
                    ->pluck('uuid');

                if ($group->type !== 'public') {
                    $undesired = $undesired->diff($members);
                }

                $notificationType->subscribers()->detach($undesired);

                if ($notificationType->roles->isNotEmpty()) {
                    $notificationType->roles()->sync($roles);
                }
            });
    }

    public function handleAdminsUpdated(Groups $originalGroup, array $oldAdmins = []): void
    {
        $group = Group::whereOriginalUuid($originalGroup->uuid)->first();

        if (! $group instanceof Group) {
            Log::error('[inside-groups] GroupsMembersService::handleAdminUpdate() : group not found in table "groups"', ['original_uuid' => $originalGroup->uuid]);
            return;
        }

        $role = Role::where('name', "group-$group->original_uuid-admin")->first();

        if (! $role instanceof Role) {
            Log::error('[inside-groups] GroupsMembersService::handleAdminUpdate() : admin role not found for group', ['original_uuid' => $originalGroup->uuid]);
            throw new \LogicException('Should not happen');
        }

        $admins = $group->admins->pluck('uuid');

        $removedAdmins = collect($oldAdmins)->diff($admins);

        if ($removedAdmins->isNotEmpty()) {
            RoleHelper::revokeUsersToRole($role->id, $removedAdmins->all());
            $removedAdmins->each(fn (string $uuid) => UserRevokedAdminEvent::dispatch($group->original, $uuid));
        }


        $newAdmins = $admins->diff($oldAdmins);

        if ($newAdmins->isNotEmpty()) {
            RoleHelper::assignUsersToRole($role->id, $newAdmins->all());
            $newAdmins->each(fn (string $uuid) => UserGrantedAdminEvent::dispatch($group->original, $uuid));
        }
    }

    public function updateMembers(Groups $originalGroup, array $newMembers): void
    {
        $group = Group::whereOriginalUuid($originalGroup->uuid)->first();

        if (! $group instanceof Group) {
            Log::error('[inside-groups] GroupsMembersService::updateMembers() : group not found in table "groups"', ['original_uuid' => $originalGroup->uuid]);
            return;
        }

        if (collect($newMembers)->diff($group->members->pluck('uuid'))->isEmpty()) {
            return;
        }
        $this->removeOldMembers($group, $newMembers);
        $this->addNewMembers($group, $newMembers);

        // Re-save to trigger events now that we have our members
        GroupsHelper::touch($originalGroup);
    }

    public function addNewMembers(Group $group, array $newMembers): void
    {
        $roleIds = $group->roles->pluck('id');

        $usersToAdd = User::query()
            ->whereNotIn('uuid', $group->members->pluck('uuid'))
            ->whereIn('uuid', $newMembers)
            ->when(
                $roleIds->isNotEmpty(),
                fn (Builder $query) => $query->whereHas('roles', fn (Builder $query) => $query->whereIn('role_id', $roleIds))
            )
            ->get();

        AddNewGroupMembers::dispatch($usersToAdd, $group, Auth::user());
    }

    public function removeOldMembers(Group $group, array $newMembers): void
    {
        $newMembers = collect($newMembers);

        $usersToRemove = $group->members->whereNotIn('uuid', $newMembers->unique());

        DeleteUsersFromGroup::dispatch($usersToRemove, $group, Auth::user());
    }
}
