<?php

namespace Inside\Groups\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Carbon;
use Inside\Database\Eloquent\Factories\HasFactory;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\GroupsPosts;
use Inside\Database\Eloquent\WithEnhancedBuilder;
use Inside\Groups\Events\GroupPostCreated;
use Inside\Permission\Scopes\AllowedScope;
use Inside\Reaction\Models\Reaction;

/**
 * @property integer $id
 * @property integer $group_id
 * @property integer|null $parent_id
 * @property string $body
 * @property string|null $file
 * @property integer $visits
 * @property-read int|null $replies_count
 * @property string $original_uuid from inside content
 * @property string $author_uuid
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read User $author
 * @property-read Group $group
 * @property-read GroupsPosts $original
 * @property-read GroupPost|null $parent
 * @property-read Collection|\Inside\Reaction\Models\Reaction[] $reactions
 * @property-read int|null $reactions_count
 * @property-read Collection|GroupPost[] $replies
 * @method static \Inside\Database\Eloquent\Builder|GroupPost newModelQuery()
 * @method static \Inside\Database\Eloquent\Builder|GroupPost newQuery()
 * @method static \Inside\Database\Eloquent\Builder|GroupPost query()
 * @method static \Illuminate\Database\Eloquent\Builder|GroupPost whereAuthorUuid($value)
 * @method static \Illuminate\Database\Eloquent\Builder|GroupPost whereBody($value)
 * @method static \Illuminate\Database\Eloquent\Builder|GroupPost whereCreatedAt($value)
 * @method static \Illuminate\Database\Eloquent\Builder|GroupPost whereFile($value)
 * @method static \Illuminate\Database\Eloquent\Builder|GroupPost whereGroupId($value)
 * @method static \Illuminate\Database\Eloquent\Builder|GroupPost whereId($value)
 * @method static \Illuminate\Database\Eloquent\Builder|GroupPost whereOriginalUuid($value)
 * @method static \Illuminate\Database\Eloquent\Builder|GroupPost whereParentId($value)
 * @method static \Illuminate\Database\Eloquent\Builder|GroupPost whereRepliesCount($value)
 * @method static \Illuminate\Database\Eloquent\Builder|GroupPost whereUpdatedAt($value)
 * @method static \Illuminate\Database\Eloquent\Builder|GroupPost whereVisits($value)
 * @mixin Collection
 * @OA\Schema (schema="Inside\Groups\Models\GroupPost",title="group_post",type="object",description="Dynamic model generated by inside content",@OA\Property(property="id",
 * title="id",
 * type="integer",
 * description=""),
 * @OA\Property (property="group_id",
 * title="group_id",
 * type="integer",
 * description=""),
 * @OA\Property (property="parent_id",
 * title="parent_id",
 * type="integer|null",
 * description=""),
 * @OA\Property (property="body",
 * title="body",
 * type="string",
 * description=""),
 * @OA\Property (property="file",
 * title="file",
 * type="string|null",
 * description=""),
 * @OA\Property (property="visits",
 * title="visits",
 * type="integer",
 * description=""),
 * @OA\Property (property="replies_count",
 * title="replies_count",
 * type="int|null",
 * description=""),
 * @OA\Property (property="original_uuid",
 * title="original_uuid",
 * type="string",
 * description=""),
 * @OA\Property (property="author_uuid",
 * title="author_uuid",
 * type="string",
 * description=""),
 * @OA\Property (property="created_at",
 * title="created_at",
 * type="\Illuminate\Support\Carbon|null",
 * description=""),
 * @OA\Property (property="updated_at",
 * title="updated_at",
 * type="\Illuminate\Support\Carbon|null",
 * description=""),
 * @OA\Property (property="author",
 * title="author",
 * type="\Inside\Authentication\Models\User",
 * description=""),
 * @OA\Property (property="group",
 * title="group",
 * type="\Inside\Groups\Models\Group",
 * description=""),
 * @OA\Property (property="original",
 * title="original",
 * type="\Inside\Content\Models\Contents\GroupsPosts",
 * description=""),
 * @OA\Property (property="parent",
 * title="parent",
 * type="\Inside\Groups\Models\GroupPost|null",
 * description=""),
 * @OA\Property (property="reactions",
 * title="reactions",
 * type="\Illuminate\Database\Eloquent\Collection|\Inside\Reaction\Models\Reaction[]",
 * description=""),
 * @OA\Property (property="reactions_count",
 * title="reactions_count",
 * type="int|null",
 * description=""),
 * @OA\Property (property="replies",
 * title="replies",
 * type="\Illuminate\Database\Eloquent\Collection|\Inside\Groups\Models\GroupPost[]",
 * description=""))
 */
class GroupPost extends Model
{
    use WithEnhancedBuilder;
    use HasFactory;

    /**
     * @var string
     */
    protected $table = 'group_posts';

    /**
     * @var array
     */
    protected $guarded = [];

    /**
     * @var array
     */
    protected $with = ['author'];

    protected static function boot(): void
    {
        parent::boot();

        static::created(
            function ($post) {
                $post->parent?->increment('replies_count');
                $post->group->update(['last_activity' => $post->created_at]);
                $post->group->increment('posts_count');
                GroupPostCreated::dispatch($post);
            }
        );

        static::deleted(
            function ($post) {
                $post->parent?->decrement('replies_count');
                $post->group->decrement('posts_count');
            }
        );
    }

    public function author(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function parent(): BelongsTo
    {
        return $this->belongsTo(GroupPost::class, 'parent_id');
    }

    public function group(): BelongsTo
    {
        return $this->belongsTo(Group::class);
    }

    public function replies(): HasMany
    {
        return $this->hasMany(GroupPost::class, 'parent_id', 'id');
    }

    public function original(): BelongsTo
    {
        return $this->belongsTo(GroupsPosts::class)->withoutGlobalScope(AllowedScope::class);
    }

    public function reactions(): HasMany
    {
        return $this->hasMany(Reaction::class, 'reactionable_uuid', 'original_uuid')->where('reactionable_type', GroupsPosts::class);
    }
}
