<?php

namespace Inside\Groups\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Carbon;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\Groups;
use Inside\Database\Eloquent\WithEnhancedBuilder;
use Inside\Permission\Models\Role;
use Inside\Permission\Scopes\AllowedScope;

/**
 * @property integer $id
 * @property string $name
 * @property string $slug
 * @property string|null $image
 * @property string|null $description
 * @property string $original_uuid from inside content
 * @property string $header_text_style
 * @property string $type
 * @property integer $enabled
 * @property string $author_uuid
 * @property Carbon|null $last_activity
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read Collection|Role[] $roles
 * @property-read Collection|User[] $admins
 * @property-read int|null $admins_count
 * @property-read User $author
 * @property-read Collection|User[] $members
 * @property-read int|null $members_count
 * @property-read Groups $original
 * @property-read Collection|User[] $pendingUsers
 * @property-read int|null $pending_users_count
 * @property-read Collection|GroupPost[] $posts
 * @property-read int|null $posts_count
 * @property-read Collection|User[] $users
 * @property-read int|null $users_count
 * @property-read Collection|User[] $nonAdmins
 * @property-read int|null $non_admins_count
 * @method static \Inside\Database\Eloquent\Builder|\Inside\Groups\Models\Group newModelQuery()
 * @method static \Inside\Database\Eloquent\Builder|\Inside\Groups\Models\Group newQuery()
 * @method static \Inside\Database\Eloquent\Builder|\Inside\Groups\Models\Group query()
 * @method static \Inside\Database\Eloquent\Builder|\Inside\Groups\Models\Group each(callable $callback, $count = 1000)
 * @method static int count($columns = '*')
 * @method static \Illuminate\Database\Eloquent\Builder|\Inside\Groups\Models\Group whereAuthorUuid($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\Inside\Groups\Models\Group whereCreatedAt($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\Inside\Groups\Models\Group whereDescription($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\Inside\Groups\Models\Group whereEnabled($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\Inside\Groups\Models\Group whereHeaderTextStyle($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\Inside\Groups\Models\Group whereId($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\Inside\Groups\Models\Group whereImage($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\Inside\Groups\Models\Group whereLastActivity($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\Inside\Groups\Models\Group whereName($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\Inside\Groups\Models\Group whereOriginalUuid($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\Inside\Groups\Models\Group whereSlug($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\Inside\Groups\Models\Group whereType($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\Inside\Groups\Models\Group whereUpdatedAt($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\Inside\Groups\Models\Group forUser($user)
 * @mixin Collection
 * @OA\Schema (schema="Inside\Groups\Models\Group",title="group",type="object",description="Dynamic model generated by inside content",@OA\Property(property="id",
 * title="id",
 * type="integer",
 * description=""),
 * @OA\Property (property="name",
 * title="name",
 * type="string",
 * description=""),
 * @OA\Property (property="slug",
 * title="slug",
 * type="string",
 * description=""),
 * @OA\Property (property="image",
 * title="image",
 * type="string|null",
 * description=""),
 * @OA\Property (property="description",
 * title="description",
 * type="string|null",
 * description=""),
 * @OA\Property (property="original_uuid",
 * title="original_uuid",
 * type="string",
 * description=""),
 * @OA\Property (property="header_text_style",
 * title="header_text_style",
 * type="string",
 * description=""),
 * @OA\Property (property="type",
 * title="type",
 * type="string",
 * description=""),
 * @OA\Property (property="enabled",
 * title="enabled",
 * type="integer",
 * description=""),
 * @OA\Property (property="author_uuid",
 * title="author_uuid",
 * type="string",
 * description=""),
 * @OA\Property (property="last_activity",
 * title="last_activity",
 * type="\Illuminate\Support\Carbon|null",
 * description=""),
 * @OA\Property (property="created_at",
 * title="created_at",
 * type="\Illuminate\Support\Carbon|null",
 * description=""),
 * @OA\Property (property="updated_at",
 * title="updated_at",
 * type="\Illuminate\Support\Carbon|null",
 * description=""),
 * @OA\Property (property="admins",
 * title="admins",
 * type="\Illuminate\Database\Eloquent\Collection|\Inside\Authentication\Models\User[]",
 * description=""),
 * @OA\Property (property="admins_count",
 * title="admins_count",
 * type="int|null",
 * description=""),
 * @OA\Property (property="author",
 * title="author",
 * type="\Inside\Authentication\Models\User",
 * description=""),
 * @OA\Property (property="members",
 * title="members",
 * type="\Illuminate\Database\Eloquent\Collection|\Inside\Authentication\Models\User[]",
 * description=""),
 * @OA\Property (property="members_count",
 * title="members_count",
 * type="int|null",
 * description=""),
 * @OA\Property (property="original",
 * title="original",
 * type="\Inside\Content\Models\Contents\Groups",
 * description=""),
 * @OA\Property (property="pendingUsers",
 * title="pendingUsers",
 * type="\Illuminate\Database\Eloquent\Collection|\Inside\Authentication\Models\User[]",
 * description=""),
 * @OA\Property (property="pending_users_count",
 * title="pending_users_count",
 * type="int|null",
 * description=""),
 * @OA\Property (property="posts",
 * title="posts",
 * type="\Illuminate\Database\Eloquent\Collection|\Inside\Groups\Models\GroupPost[]",
 * description=""),
 * @OA\Property (property="posts_count",
 * title="posts_count",
 * type="int|null",
 * description=""),
 * @OA\Property (property="users",
 * title="users",
 * type="\Illuminate\Database\Eloquent\Collection|\Inside\Authentication\Models\User[]",
 * description=""),
 * @OA\Property (property="users_count",
 * title="users_count",
 * type="int|null",
 * description=""),
 * @OA\Property (property="non_admins",
 * title="non_admins",
 * type="\Illuminate\Database\Eloquent\Collection|\Inside\Authentication\Models\User[]",
 * description=""),
 * @OA\Property (property="non_admins_count",
 * title="non_admins_count",
 * type="int|null",
 * description=""))
 *
 * @method static Builder|Group wherePostsCount($value)
 */
class Group extends Model
{
    use WithEnhancedBuilder;

    /**
     * @var string
     */
    protected $table = 'groups';

    /**
     * @var array
     */
    protected $guarded = [];

    /**
     * @var array
     */
    protected $with = ['author'];

    /**
     * @var array
     */
    protected $casts = [
        'enabled' => 'boolean',
    ];

    /**
     * @var array
     */
    protected $dates = [
        'last_activity',
    ];

    /**
     * @return void
     */
    protected static function boot(): void
    {
        parent::boot();

        static::deleting(
            function (Group $group) {
                $group->posts->each->delete();
            }
        );
    }

    public function canUserAccess(User $user): bool
    {
        if ($user->isSuperAdmin()) {
            return true;
        }

        return match ($this->type) {
            'public' => $this->roles->isEmpty() || $user->hasAnyRole($this->roles),
            default => $this->members->contains($user),
        };
    }

    public function author(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function posts(): HasMany
    {
        return $this->hasMany(GroupPost::class);
    }

    public function original(): BelongsTo
    {
        return $this->belongsTo(Groups::class)->withoutGlobalScope(AllowedScope::class);
    }

    public function users(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'group_members')->where('inside_users.status', '1');
    }

    public function roles(): BelongsToMany
    {
        return $this->belongsToMany(Role::class, 'group_roles');
    }

    public function members(): BelongsToMany
    {
        return $this->users()->wherePivotIn('type', ['admin', 'member']);
    }

    public function admins(): BelongsToMany
    {
        return $this->users()->wherePivot('type', 'admin');
    }

    public function pendingUsers(): BelongsToMany
    {
        return $this->users()->wherePivot('type', 'pending');
    }

    public function nonAdmins(): BelongsToMany
    {
        return $this->users()->wherePivot('type', 'member');
    }

    public function scopeForUser(Builder $query, User $user): void
    {
        if ($user->isSuperAdmin()) {
            return;
        }

        $query->where(
            fn (Builder $query) => $query
                ->whereHas('members', fn (Builder $query) => $query->where('user_uuid', $user->uuid))
                ->orWhere(
                    fn (Builder $query) => $query
                        ->whereIn('groups.type', ['public', 'restricted'])
                        ->where(
                            fn (Builder $query) => $query
                                ->whereDoesntHave('roles')
                                ->orWhereHas('roles', fn (Builder $query) => $query->whereIn('role_id', $user->getRoleIds()))
                        )
                )
        );
    }
}
