<?php

declare(strict_types=1);

namespace Inside\Groups\Listeners;

use Illuminate\Events\Dispatcher;
use Illuminate\Support\Facades\Log;
use Inside\Content\Events\ContentDeletedEvent;
use Inside\Content\Events\ContentFullyInsertedEvent;
use Inside\Content\Events\ContentFullyUpdatedEvent;
use Inside\Content\Models\Contents\Groups;
use Inside\Content\Models\Contents\Users;
use Inside\Groups\Events\UserJoinedAGroupEvent;
use Inside\Groups\Events\UserLeftAGroupEvent;
use Inside\Groups\Facades\GroupsMembers;
use Inside\Groups\Models\Group;
use Throwable;

/**
 * @package Inside\Groups\Listeners
 */
class SynchronizeGroup
{
    /**
     * @param ContentFullyInsertedEvent $event
     * @return void
     */
    public function handleGroupCreation(ContentFullyInsertedEvent $event): void
    {
        if (! $event->model instanceof Groups) {
            return;
        }
        $groupContent = $event->model;
        try {
            Log::debug('[inside-groups]{handleGroupCreation} sync group <'.$groupContent->uuid.'> on creation');
            $group = Group::create(
                [
                    'enabled' => $groupContent->status,
                    'name' => $groupContent->title,
                    'slug' => $groupContent->slug[0],
                    'description' => $groupContent->introduction,
                    'original_uuid' => $groupContent->uuid,
                    'image' => $groupContent->image,
                    'type' => $groupContent->visibility,
                    'author_uuid' => $groupContent->author,
                ]
            );
            self::syncMembers($groupContent, $group);
            self::syncRoles($groupContent, $group);
        } catch (Throwable $e) {
            Log::error('[groups] Failed to sync group ( creation ) => '.$e->getMessage());
        }
    }

    /**
     * @param Groups $originalGroup
     * @param Group $group
     * @return void
     */
    public function syncMembers(Groups $originalGroup, Group $group): void
    {
        $members = $originalGroup->belongsToMany(
            Users::class,
            'inside_pivots',
            'related_uuid',
            'parent_uuid'
        )->where('related_field', 'groups')->pluck('uuid')->toArray();

        $admins = $originalGroup->members->pluck('uuid')->toArray();
        $pendings = $originalGroup->pendingMembers->pluck('uuid')->toArray();
        $toSync = [];
        foreach ($members as $member) {
            $type = 'member';
            if (in_array($member, $admins)) {
                $type = 'admin';
            }
            $toSync[$member] = ['type' => $type];
        }
        foreach ($pendings as $pending) {
            $toSync[$pending] = ['type' => 'pending'];
        }
        Log::debug('[inside-groups] sync group members <'.$group->id.'> => '.json_encode($toSync));
        $group->users()->sync($toSync);
    }

    public function syncRoles(Groups $originalGroup, Group $group): void
    {
        $roleIds = str($originalGroup->groups_roles ?? '')
            ->explode(',')
            ->filter()
            ->map(fn (string $item) => intval($item))
            ->all();

        Log::debug('[inside-groups] sync group roles <'.$group->id.'> => '.json_encode($roleIds));
        $group->roles()->sync($roleIds);
        // needs to sync notifications_types if roles changes
        GroupsMembers::unsubscribeUndesiredSubscribers($originalGroup);
    }

    /**
     * @param ContentFullyUpdatedEvent $event
     * @return void
     */
    public function handleGroupModification(ContentFullyUpdatedEvent $event): void
    {
        if (! $event->model instanceof Groups) {
            return;
        }
        $groupContent = $event->model;
        try {
            Log::debug(
                '[inside-groups]{handleGroupModification} sync group <'.$groupContent->uuid.'> on modification'
            );
            $group = Group::where('original_uuid', $groupContent->uuid)->firstOrFail();
            $group->update(
                [
                    'enabled' => $groupContent->status,
                    'name' => $groupContent->title,
                    'slug' => $groupContent->slug[0],
                    'description' => $groupContent->introduction,
                    'image' => $groupContent->image,
                    'type' => $groupContent->visibility,
                ]
            );
            self::syncMembers($groupContent, $group);
            self::syncRoles($groupContent, $group);
        } catch (Throwable $e) {
            Log::error('[groups] Failed to sync group ( modification ) => '.$e->getMessage());
        }
    }

    /**
     * @param ContentDeletedEvent $event
     * @return void
     */
    public function handleGroupDeletion(ContentDeletedEvent $event): void
    {
        if (class_to_type($event->model) !== 'groups') {
            return;
        }
        try {
            $group = Group::find($event->model->uuid);
            if ($group !== null) {
                $group->delete();
            }
        } catch (Throwable $e) {
            Log::error('[groups] Failed to sync group ( deletion ) => '.$e->getMessage());
        }
    }

    /**
     * @param UserJoinedAGroupEvent $event
     * @return void
     */
    public function handleNewMember(UserJoinedAGroupEvent $event): void
    {
        $groupContent = $event->group;
        try {
            Log::debug(
                '[inside-groups]{handleNewMember} a user ['.$event->user->uuid.'] joined the group <'
                .$groupContent->uuid.'>'
            );
            $group = Group::where('original_uuid', $groupContent->uuid)->firstOrFail();
            $members = $groupContent->belongsToMany(
                Users::class,
                'inside_pivots',
                'related_uuid',
                'parent_uuid'
            )->where('related_field', 'groups')->pluck('uuid')->toArray();

            $admins = $groupContent->members->pluck('uuid')->toArray();
            $pendings = $groupContent->pendingMembers->pluck('uuid')->toArray();
            $toSync = [];
            foreach ($members as $member) {
                $type = 'member';
                if (in_array($member, $admins)) {
                    $type = 'admin';
                }
                $toSync[$member] = ['type' => $type];
            }
            foreach ($pendings as $pending) {
                $toSync[$pending] = ['type' => 'pending'];
            }
            $toSync[$event->user->uuid] = ['type' => 'member'];
            Log::debug('[inside-groups] sync group members <'.$group->id.'> => '.json_encode($toSync));

            $group->users()->sync($toSync);
        } catch (Throwable $e) {
            Log::error('[groups] Failed to handleNewMember => '.$e->getMessage());
        }
    }

    /**
     * @param UserLeftAGroupEvent $event
     * @return void
     */
    public function handleRemovedMember(UserLeftAGroupEvent $event): void
    {
        $groupContent = $event->group;
        try {
            Log::debug(
                '[inside-groups]{handleRemovedMember} a user ['.$event->user->uuid.'] left the group <'
                .$groupContent->uuid.'>'
            );
            $group = Group::where('original_uuid', $groupContent->uuid)->firstOrFail();

            $members = $groupContent->belongsToMany(
                Users::class,
                'inside_pivots',
                'related_uuid',
                'parent_uuid'
            )->where('related_field', 'groups')->pluck('uuid')->toArray();

            $admins = $groupContent->members->pluck('uuid')->toArray();
            $pendings = $groupContent->pendingMembers->pluck('uuid')->toArray();
            $toSync = [];
            foreach ($members as $member) {
                if ($member == $event->user->uuid) {
                    continue;
                }
                $type = 'member';
                if (in_array($member, $admins)) {
                    $type = 'admin';
                }
                $toSync[$member] = ['type' => $type];
            }
            foreach ($pendings as $pending) {
                $toSync[$pending] = ['type' => 'pending'];
            }
            Log::debug('[inside-groups] sync group members <'.$group->id.'> => '.json_encode($toSync));

            $group->users()->sync($toSync);
        } catch (Throwable $e) {
            Log::error('[groups] Failed to handleNewMember => '.$e->getMessage());
        }
    }

    public function subscribe(Dispatcher $events): array
    {
        return [
            ContentFullyInsertedEvent::class => 'handleGroupCreation',
            ContentFullyUpdatedEvent::class => 'handleGroupModification',
            ContentDeletedEvent::class => 'handleGroupDeletion',
            UserJoinedAGroupEvent::class => 'handleNewMember',
            UserLeftAGroupEvent::class => 'handleRemovedMember',
        ];
    }
}
