<?php

namespace Inside\Groups\Http\Middlewares;

use Closure;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\Groups;
use Inside\Groups\Events\UserGrantedAdminEvent;
use Inside\Groups\Events\UserJoinedAGroupEvent;
use Inside\Groups\Events\UserLeftAGroupEvent;
use Inside\Groups\Events\UserRevokedAdminEvent;
use Inside\Groups\Facades\GroupsHelper;
use Inside\Groups\Facades\GroupsMembers;
use Inside\Notify\Models\NotificationSubscriber;
use Inside\Notify\Models\NotificationType;
use Inside\Permission\Facades\Role;
use Inside\Permission\Models\Role as InsideRole;
use Symfony\Component\HttpFoundation\Response;

class MembersFieldMiddleware
{
    /**
     * @throws Exception
     */
    public function handle(Request $request, Closure $next): mixed
    {
        $path = $request->path();
        $method = $request->getMethod();
        $pattern = '#^api/v1/form/groups/[\da-f]{8}-[\da-f]{4}-[\da-f]{4}-[\da-f]{4}-[\da-f]{12}$#';

        if (preg_match($pattern, $path)) {
            return $this->handleForm($request, $next);
        }

        $pattern = '#^api/v1/content/groups/[\da-f]{8}-[\da-f]{4}-[\da-f]{4}-[\da-f]{4}-[\da-f]{12}$#';
        if (preg_match($pattern, $path) && $method == 'PUT') {
            return $this->handleGroups($request, $next);
        }

        if (preg_match('#^api/v1/content/groups$#', $path) && $method == 'POST') {
            return $this->handleNewGroup($request, $next);
        }

        return $next($request);
    }

    protected function handleForm(Request $request, Closure $next): mixed
    {
        $response = $next($request);
        if ($response instanceof Response && $response->getStatusCode() != 200) {
            return $response;
        }
        $memberField = false;
        $data = json_decode_response($response);
        if ($data == null || !isset($data['data']) || !is_array($data['data'])) {
            return $response;
        }

        foreach ($data['data'] as $key => &$value) {
            if (isset($value['name']) && $value['name'] === 'members') {
                $memberField = &$value;
                break;
            }
            if (!isset($value['fields'])) {
                continue;
            }
            if (is_array($value['fields'])) {
                foreach ($value['fields'] as $fieldKey => &$fieldValue) {
                    if (isset($fieldValue['name']) && $fieldValue['name'] === 'members') {
                        $memberField = &$fieldValue;
                        break;
                    }
                }
            }
        }

        if (!$memberField) {
            return $response;
        }

        $matches = [];
        $pattern = '#^api/v1/form/groups/([\da-f]{8}-[\da-f]{4}-[\da-f]{4}-[\da-f]{4}-[\da-f]{12})$#';
        preg_match($pattern, $request->path(), $matches);
        if (count($matches) !== 2) {
            return $response;
        }
        $group = Groups::find($matches[1]);
        if (!$group) {
            abort(500, 'system error' . (env('APP_DEBUG', false) ? ' [MembersFieldMiddleware::handleForm]' : ''));
        }
        $values = [];
        foreach ($group->reverseUsers as $member) {
            $values[] = [
                'uuid' => $member->uuid,
                'admin' => $group->members->contains($member),
            ];
        }
        $memberField['value'] = $values;
        set_response($response, $data);

        return $response;
    }

    /**
     * @throws Exception
     */
    protected function handleGroups(Request $request, Closure $next): mixed
    {
        $path = $request->path();
        $matches = [];

        preg_match(
            '#^api/v1/content/groups/([\da-f]{8}-[\da-f]{4}-[\da-f]{4}-[\da-f]{4}-[\da-f]{12})$#',
            $path,
            $matches
        );

        if (count($matches) != 2 && $request->getMethod() != 'PUT') {
            return $next($request);
        }

        $group = Groups::find($matches[1]);
        if (!$group) {
            abort(500, 'system error' . (env('APP_DEBUG', false) ? ' [MembersFieldMiddleware::handleGroups]' : ''));
        }

        $oldAdmins = $group->members->pluck('uuid');

        $requestMembers = collect($request->get('members', []));

        $members = $requestMembers->pluck('uuid');
        $admins = $requestMembers->filter(fn ($member) => $member['admin'])->pluck('uuid');

        $request['members'] = $admins->all();

        // we remove new users from pending_members
        if ($group->visibility === 'restricted' && $request->has('pending_members')) {
            $pendingMembers = collect($request->get('pending_members', []))->diff($members);
            $request['pending_members'] = $pendingMembers->all();
        }

        $response = $next($request);
        if (!$response instanceof Response) {
            return $response;
        }

        GroupsMembers::removeUndesiredGroupMembers($group);
        GroupsMembers::handleAdminsUpdated($group, $oldAdmins->all());
        GroupsMembers::updateMembers($group, $members->all());
        GroupsMembers::unsubscribeUndesiredSubscribers($group);

        return $response;
    }

    protected function handleNewGroup(Request $request, Closure $next): mixed
    {
        $me = Auth::user();
        /** @var User $me */
        if ($me == null) {
            return $next($request);
        }
        $request['members'] = [$me->uuid];

        return $next($request);
    }
}
