<?php

namespace Inside\Groups\Http\Middlewares;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\Groups;
use Inside\Permission\Facades\Role as RoleService;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;

class GroupsRolesMiddleware
{
    public function handle(Request $request, Closure $next): mixed
    {
        $path = $request->path();
        if (Str::startsWith($request->path(), 'api/v1/form/groups')) {
            return $this->handleGroupsForm($request, $next);
        }

        if (
            Str::startsWith($request->path(), 'api/v1/content/groups')
            && in_array($request->method(), ['POST', 'PUT'])
        ) {
            return $this->handleCreateOrUpdateGroups($request, $next);
        }

        return $next($request);
    }

    private function getContentFromPath(string $path): ?Groups
    {
        if (! Str::isUuid(Str::afterLast($path, '/'))) {
            return null;
        }

        return Groups::find(Str::afterLast($path, '/'));
    }

    /**
     * Function that transform the coma separated list groups_roles into an array of ids
     * in order to be able to use the SELECT_WIDGET
     */
    private function handleGroupsForm(Request $request, Closure $next): mixed
    {
        $response = $next($request);

        if ($response instanceof Response && $response->getStatusCode() !== 200) {
            return $response;
        }

        $data = json_decode_response($response);

        $allowedValues = collect(list_languages())
            ->mapWithKeys(fn (string $language) => [$language => $this->getRolesAllowedValues($language)])
            ->all();

        $group = $this->getContentFromPath($request->path());

        $value = str($group?->groups_roles ?? '')
            ->explode(',')
            ->filter()
            ->map(fn (string $roleId) => intval($roleId))
            ->all();

        foreach ($data['data'] as &$field) {
            if (isset($field['name']) && $field['name'] === 'groups_roles') {
                $field['options']['allowed_values'] = [
                    'roles' => $allowedValues,
                ];

                $field['options']['cardinality'] = -1;
                $field['options']['default'] = [];
                $field['options']['widget'] = 'selectroles';
                $field['value'] = $value;
            }
        }

        set_response($response, $data);

        return $response;
    }

    /**
     * Function that transform the array of ids groups_roles to a coma separated list of id
     * in order to store the list in the content
     */
    public function handleCreateOrUpdateGroups(Request $request, Closure $next): mixed
    {
        /** @var User $user */
        $user = Auth::user();

        if ($request->has('groups_roles')) {
            $inputs = $request->all();
            $roles = $inputs['groups_roles'];

            if (! empty($roles) && ! $user->isSuperAdmin() && ! $user->hasAnyRole($roles)) {
                throw new BadRequestHttpException(Lang::get('groups.form.groups_roles.error'));
            }

            $inputs['groups_roles'] = collect($roles)->implode(',') ?: null;
            $request->replace($inputs);
        }

        return $next($request);
    }

    private function getRolesAllowedValues(string $locale): array
    {
        $roles = RoleService::tree();
        $allowedValues = [];
        foreach ($roles as $role) {
            $allowedValue = [
                'title' => RoleService::getHumanName($role['name'], $locale),
                'uuid' => $role['id'] + ($role['type'] === 'role' ? 0 : 10000), // avoid id collision
                'type' => $role['type'],
                'children' => [],
            ];

            foreach ($role['children'] as $roleChildren) {
                $roleInformation = [
                    'title' => RoleService::getHumanName($roleChildren['name'], $locale),
                    // avoid id collision
                    'uuid' => $roleChildren['id'] + ($roleChildren['type'] === 'role' ? 0 : 10000),
                    'type' => $roleChildren['type'],
                ];
                $allowedValue['children'][] = $roleInformation;
            }
            $allowedValues[] = $allowedValue;
        }

        return $allowedValues;
    }
}
