<?php

declare(strict_types=1);

namespace Inside\Groups\Http\Controllers;

use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inside\Authentication\Models\User;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Models\Contents\Groups;
use Inside\Content\Models\Contents\Users;
use Inside\Groups\Events\UserAskedToJoinAGroupEvent;
use Inside\Groups\Models\Group;
use Inside\Groups\Repositories\GroupPostRepository;
use Inside\Groups\Repositories\GroupRepository;
use Inside\Host\Bridge\BridgeContent;
use Inside\Notify\Models\NotificationType;
use Inside\Permission\Exceptions\AuthorizationException;
use Inside\Support\Str;
use InvalidArgumentException;
use Laravel\Lumen\Routing\Controller;
use OpenApi\Annotations as OA;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * @OA\Tag(name="Groupes",description="
# Introduction
Il s'agit du module de groupes/communautés d'Inside 2.
")
 *
 * @package Inside\Groups\Http\Controllers
 */
final class GroupsController extends Controller
{
    public function __construct(
        protected GroupPostRepository $groupPostRepository,
        protected GroupRepository $groupRepository
    ) {
    }

    public function members(string $uuid, Request $request): JsonResponse
    {
        $group = Group::where('original_uuid', $uuid)->firstOrFail();
        $usersQuery = Users::query()->whereIn('uuid', $group->members()->pluck('uuid')->all());
        $filters = ContentHelper::extractFiltersInputFromRequest($request);

        return response()->json(
            ContentHelper::getResultFromQuery($usersQuery, $filters)
        );
    }

    /**
     * @throws Exception
     */
    public function askToJoin(string $uuid): JsonResponse
    {
        $group = Group::whereOriginalUuid($uuid)->first();

        /** @var ?User $me */
        $me = Auth::user();

        if (! $me instanceof User) {
            throw new AccessDeniedHttpException('forbidden');
        }

        if (! $group instanceof Group || ! $group->original instanceof Groups) {
            throw new NotFoundHttpException();
        }

        if ($group->type !== 'restricted') {
            throw new InvalidArgumentException('this group is not a restricted group');
        }

        if ($group->roles->isNotEmpty() && ! $me->hasAnyRole($group->roles->all())) {
            throw AuthorizationException::create('read', 'groups', $uuid);
        }

        if ($group->users->contains($me)) {
            return response()->json(
                [
                    'already_member' => $group->members->contains($me),
                    'already_pending' => $group->pendingUsers->contains($me),
                ]
            );
        }

        $pendingUsers = $group->pendingUsers->pluck('uuid')->push($me->uuid)->unique()->all();

        $bridge = new BridgeContent();
        $bridge->contentUpdate(
            'groups',
            [
                'uuid' => $group->original_uuid,
                'pending_members' => $pendingUsers,
            ]
        );

        UserAskedToJoinAGroupEvent::dispatch($group->original, $me);

        return response()->json(
            [
                'asked' => true,
            ]
        );
    }

    public function isMember(string $uuid): JsonResponse
    {
        $group = Group::whereOriginalUuid($uuid)->first();

        $me = Auth::user();

        if (! $me instanceof User) {
            throw new AccessDeniedHttpException('forbidden');
        }

        if (! $group instanceof Group) {
            throw new NotFoundHttpException();
        }

        return response()->json(
            [
                'is_member' => $group->members->contains($me),
            ]
        );
    }

    public function isAdmin(string $uuid): JsonResponse
    {
        $group = Group::whereOriginalUuid($uuid)->first();

        $me = Auth::user();

        if (! $me instanceof User) {
            throw new AccessDeniedHttpException('forbidden');
        }

        if (! $group instanceof Group) {
            throw new NotFoundHttpException();
        }

        return response()->json(
            [
                'admin' => $group->admins->contains($me),
            ]
        );
    }

    public function isSubscribed(string $uuid): JsonResponse
    {
        /** @var ?User $user */
        $user = Auth::user();

        if (! $user instanceof User) {
            throw new AccessDeniedHttpException('forbidden');
        }

        $isSubscribed = NotificationType::query()
            ->where([
                'action' => 'custom',
                'condition' => "groups:$uuid",
            ])
            ->whereHas('subscribers', fn (Builder $query) => $query->where('user_uuid', $user->uuid))
            ->exists();

        return response()->json(
            [
                'isSubscribed' => $isSubscribed,
            ]
        );
    }

    public function subscribe(string $uuid): JsonResponse
    {
        $user = Auth::user();

        if (! $user instanceof User) {
            throw new AccessDeniedHttpException('forbidden');
        }

        NotificationType::query()
            ->where('condition', 'groups:'.$uuid)
            ->where('action', 'custom')
            ->each(function (NotificationType $type) use ($user) {
                if (! $type->subscribers->contains($user->uuid)) {
                    $type->subscribers()->attach($user->uuid);
                }
            });

        return response()->json(['subscribed' => true]);
    }

    /**
     * @throws Exception
     */
    public function unsubscribe(string $uuid): JsonResponse
    {
        $user = Auth::user();

        if (! $user instanceof User) {
            throw new AccessDeniedHttpException('forbidden');
        }

        NotificationType::query()
            ->where('condition', 'groups:'.$uuid)
            ->where('action', 'custom')
            ->each(function (NotificationType $type) use ($user) {
                $type->subscribers()->detach($user->uuid);
            });

        return response()->json(['unsubscribed' => true]);
    }

    /**
     * @OA\Get(
     *      path="/groups/posts/{group}/index",
     *      operationId="getGroupPostsList",
     *      tags={"Groupes"},
     *      summary="Lister les messages du groupe $group",
     *     @OA\Parameter(in="path",name="group",description="Uuid du groupe ou son
     *                                                            slug",required=true,type="string",example="mon-groupe"),
     *     @OA\Parameter (in="query",name="query",description="Paramètre de recherche.",type="string",example="mot1
    mot2"),
     *     @OA\Parameter (in="query",name="limit",optional=true,description="Limiter le nombre de
     *                   groupes.",type="integer",example=5),
     *     @OA\Parameter (in="query",name="page",optional=true,description="Demander la page en
     *                   question",type="integer",example=2),
     *     @OA\Response(
     *          response=200,
     *          description="Opération réussie",
     *          @OA\JsonContent(ref="#/components/schemas/GroupPostList")
     *       ),
     *      @OA\Response(
     *          response=401,
     *          description="Non connecté",
     *      ),
     *      @OA\Response(
     *          response=403,
     *          description="Opération interdite"
     *      )
     *     )
     *
     */
    public function posts(string $uuid, Request $request): JsonResponse
    {
        $user = Auth::user();

        if (Str::isUuid($uuid)) {
            $group = Group::where('original_uuid', $uuid)->first();
        } else {
            $group = Group::where('slug', $uuid)->first();
        }

        if (! $group instanceof Group) {
            throw new NotFoundHttpException();
        }

        if (! $user instanceof User || ! $group->canUserAccess($user)) {
            throw AuthorizationException::create('read', 'groups', $uuid);
        }

        $filters = ContentHelper::extractFiltersInputFromRequest($request);

        return response()->json(
            $this->groupPostRepository->list(
                $group,
                $request->get('query'),
                $filters['limit'] ?? null,
                isset($filters['paginate']) && $filters['paginate']
            )
        );
    }

    /**
     * @OA\Get(
     *      path="/groups/posts/{post}",
     *      operationId="getGroupsPost",
     *      tags={"Groupes"},
     *      summary="Récupérer les informations d'un message de groupe.",
     *     @OA\Parameter (in="query",name="query",description="Paramètre de recherche.",type="string",example="mot1
     *                   mot2"),
     *     @OA\Response(
     *          response=200,
     *          description="Opération réussie",
     *          @OA\JsonContent(ref="#/components/schemas/GroupResource")
     *       ),
     *      @OA\Response(
     *          response=401,
     *          description="Non connecté",
     *      ),
     *      @OA\Response(
     *          response=403,
     *          description="Opération interdite"
     *      )
     *     )
     *
     */
    public function post(string $uuid, Request $request): JsonResponse
    {
        return response()->json($this->groupPostRepository->get($uuid, $request->get('query')));
    }

    /**
     * @OA\Get(
     *      path="/groups/index",
     *      operationId="getGroupsList",
     *      tags={"Groupes"},
     *      summary="Lister les groupes disponibles. Sans pagination, trié par date de dernière activité puis
     *      alphabétique.",
     *     @OA\Parameter (in="query",name="query",description="Paramètre de recherche.",type="string",example="mot1
     *                   mot2"),
     *     @OA\Response(
     *          response=200,
     *          description="Opération réussie",
     *          @OA\JsonContent(ref="#/components/schemas/GroupList")
     *       ),
     *      @OA\Response(
     *          response=401,
     *          description="Non connecté",
     *      ),
     *      @OA\Response(
     *          response=403,
     *          description="Opération interdite"
     *      )
     *     )
     *
     */
    public function groups(Request $request): JsonResponse
    {
        $filters = ContentHelper::extractFiltersInputFromRequest($request);

        return response()->json(
            $this->groupRepository->list(
                $request->get('query', null),
                $filters['limit'] ?? null
            )
        );
    }

    /**
     * @OA\Get(
     *      path="/groups/list",
     *      operationId="getGroupsSimpleList",
     *      tags={"Groupes"},
     *      summary="Enumérer les groupes ( nom, type ).",
     *     @OA\Parameter (in="query",name="limit",optional=true,description="Limiter le nombre de
     *                   groupes.",type="integer",example=5),
     *     @OA\Response(
     *          response=200,
     *          description="Opération réussie",
     *          @OA\JsonContent(ref="#/components/schemas/SimpleGroupList")
     *       ),
     *      @OA\Response(
     *          response=401,
     *          description="Non connecté",
     *      ),
     *      @OA\Response(
     *          response=403,
     *          description="Opération interdite"
     *      )
     *     )
     *
     */
    public function simpleListGroups(Request $request): JsonResponse
    {
        return response()->json(
            $this->groupRepository->enumerate($request->get('limit'))
        );
    }
}
