<?php

namespace Inside\Form\Services;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Date;
use Illuminate\Support\Facades\Storage;
use Inside\Content\Models\Content;
use Inside\Content\Models\Contents\Forms;
use Inside\Content\Models\Contents\Users;
use Inside\Form\Events\FormSubmissionDeletedEvent;
use Inside\Form\Events\FormSubmissionStatusEditedEvent;
use Inside\Form\Events\FormSubmitedEvent;
use Inside\Form\Facades\FormAnswers;
use Inside\Form\Facades\FormFile;
use Inside\Form\Facades\FormUsers;
use Inside\Form\Facades\SurveyAnswers;
use Inside\Form\Models\FormAnswer;
use Inside\Form\Models\FormSubmission;

class FormService
{
    private function getAnswerFile(?string $file, string $fieldId, FormSubmission $formSubmission): ?array
    {
        $formAnswer = $formSubmission->answers->firstWhere('field_id', $fieldId);

        if ($formAnswer instanceof FormAnswer && isset($formAnswer->answer['file'])) {
            if ($file === $formAnswer->answer['file']) {
                return $formAnswer->answer;
            }

            FormFile::deleteFile($formAnswer->answer['file']);
        }

        if (empty($file)) {
            return null;
        }

        return FormFile::saveFile($formSubmission->id, $formSubmission->answerable_uuid, $file);
    }

    private function formatAnswers(array $answers, FormSubmission $formSubmission): array
    {
        $formAnswers = [];

        foreach ($answers as $answer) {
            if ($answer['field_type'] === 'input-attachment') {
                $answer['value'] = $this->getAnswerFile($answer['value'], $answer['field_id'], $formSubmission) ?? '';
            }

            if ($answer['field_type'] === 'input-date' && is_string($answer['value']) && ! empty($answer['value'])) {
                $answer['value'] = strtotime($answer['value']);
            }

            $formAnswers[] = [
                'field_id' => $answer['field_id'],
                'field_type' => $answer['field_type'],
                'answer' => $answer['value'],
            ];
        }

        return $formAnswers;
    }

    private function formatUser(?string $userUuid, bool $isAdmin = false): ?array
    {
        $user = Users::find($userUuid);

        if (! $user instanceof Users) {
            return null;
        }

        return [
            'uuid' => $user->uuid,
            'firstname' => $user->firstname,
            'lastname' => $user->lastname,
            'isAdmin' => $isAdmin,
        ];
    }

    public function storeAnswers(array $data, array $answers = []): void
    {
        $answerableUuid = $data['answerable_uuid'];
        $userUuid = $data['user_uuid'];

        $formSubmission = FormSubmission::create(
            [
                'answerable_uuid' => $answerableUuid,
                'answerable_type' => $data['answerable_type'],
                'user_uuid' => $userUuid,
            ]
        );

        $formAnswers = $this->formatAnswers($answers, $formSubmission);
        $formSubmission->answers()->createMany($formAnswers);

        event(new FormSubmitedEvent($formSubmission));
    }

    public function updateSubmissionAnswers(FormSubmission $formSubmission, array $answers, string $editorUuid): void
    {
        $formAnswers = $this->formatAnswers($answers, $formSubmission);

        $formSubmission->answers()->delete();
        $formSubmission->answers()->createMany($formAnswers);

        $formSubmission->edited_at = Date::now();
        $formSubmission->editor_uuid = $editorUuid;
        $formSubmission->status = FormSubmission::DEFAULT_STATUS;
        $formSubmission->save();

        event(new FormSubmitedEvent($formSubmission));
    }

    /**
     * @param string $related
     * @param bool $forceSurvey
     * @param bool $trending
     * @param bool $myAnswers
     * @param bool $export
     * @return array|null
     */
    public function getAnswers(
        string $related,
        bool $forceSurvey = false,
        bool $trending = false,
        bool $myAnswers = true,
        bool $export = false,
        bool $paginate = false,
        int $paginationLimit = 0
    ): ?array {
        $relatedTypeClass = 'Inside\Content\Models\Contents\Forms';

        $query = call_user_func($relatedTypeClass.'::query');
        $content = $query->find($related);
        $currentUser = Auth::user()->uuid ?? FormUsers::getUserAuthenticatedForJob()?->uuid;
        $admin = ! $myAnswers && FormUsers::isAdmin($related);
        $user = $admin ? null : $currentUser;

        $results = [];

        if ($content->form_type === 'forms' && ! $forceSurvey) {
            $results = FormAnswers::getFormAnswers($content, $trending, $admin, $export, $paginate, $paginationLimit);
        } else {
            if ($content->form_type === 'surveys' || $forceSurvey) {
                // We can't filter by user for a survey
                $user = null;
                $results = SurveyAnswers::getSurveyAnswers($content);
            }
        }

        $contributors = FormUsers::getContributors($related);

        return array_merge(
            $results,
            [
                'counter' => $this->getAnswersCount($related, $user),
                'contributors' => $contributors,
                'user' => $this->formatUser($user, FormUsers::isAdmin($related)),
                'isAdmin' => FormUsers::isAdmin($related),
            ]);
    }

    public function getSubmission(string $related, int $id): ?array
    {
        $content = Forms::find($related);
        $fields = json_decode($content->inputs);

        $formatedResults = [];

        foreach ($fields as $field) {
            $formatedResults[] = [
                'field_id' => $field->field_id,
                'field_order' => $field->field_order,
                'field_question' => $field->field_question,
                'field_choices' => $field->field_choices,
                'field_type' => $field->field_type,
                'field_closed_ended' => $field->field_closed_ended,
                'field_user_can' => $field->show_in_list_user ?? $field->field_user_can->show_in_modal_user ?? false,
            ];
        }

        $submission = FormSubmission::find($id);

        if (empty($submission)) {
            return []; // throw une 404 ?
        }

        $isAdmin = FormUsers::isAdmin($related);

        $data = [
            'common' => [
                'uuid' => $submission->id,
                'request_date' => $submission->created_at,
                'edit_date' => $submission->edited_at,
                'status' => $submission->status,
                'comment' => $submission->comment,
                'code' => $submission->code,
            ],
            'user' => $this->formatUser($submission->user_uuid, $isAdmin),
            'editor' => $this->formatUser($submission->editor_uuid),
            'isAdmin' => $isAdmin,
            'submission' => [],
            'anonym' => $content->anonym,
        ];

        foreach ($formatedResults as $object) {
            if (! $isAdmin && ! $object['field_user_can']) {
                continue;
            }

            $answer = [
                'field_id' => $object['field_id'],
                'response' => null,
            ];

            foreach ($submission->answers as $submissionAnswer) {
                if ($submissionAnswer->field_id === $object['field_id']) {
                    $answer['response'] = $submissionAnswer->answer;
                    if ($object['field_type'] === 'input-attachment') {
                        $answer = $this->prepareFileFieldAnswer($content, $answer);
                    }
                }
            }

            $data['submission'][] = $answer;
        }

        if ($content->anonym) {
            unset($data['user']['firstname'], $data['user']['lastname']);
        }

        return $data;
    }

    /**
     * @param Content $content
     * @param array $answer
     */
    public function prepareFileFieldAnswer(Content $content, array $answer): array
    {
        if (
            is_array($answer['response'])
            && ! empty($answer['response'])
            && isset($answer['response']['file'])
            && ! empty($answer['response']['file'])
        ) {
            $file = $answer['response']['file'];
            $disk = Storage::disk('local');
            if ($disk->exists($file)) {
                $answer['options'] = [
                    'label' => $answer['response']['name'] ?? 'Fichier',
                    'mimetype' => mime_content_type($disk->path($file)),
                    'file' => $file,
                ];

                $answer['response'] = $disk->url((data_get($content, 'slug.0') ?? ($content->uuid.'/forms')).'/'.$file);
            } else {
                $answer['response'] = '';
            }
        }

        return $answer;
    }

    public function editSubmissionStatus(int $status, int $id, bool $notify, ?string $comment): ?FormSubmission
    {
        $submission = FormSubmission::find($id);
        if (empty($submission)) {
            return null;
        }

        $isAnonym = Forms::find($submission->answerable_uuid)->anonym;

        $submission->fill([
            'notify' => $notify,
            'comment' => $comment,
            'status' => $status,
        ]);
        $submission->save();
        $submission->anonym = $isAnonym;

        event(new FormSubmissionStatusEditedEvent($submission));

        return $submission;
    }

    public function getAnswersCount(string $related, string $user = null): ?int
    {
        $query = FormSubmission::where('answerable_uuid', $related);

        if ($user) {
            $query->where('user_uuid', $user);
        }

        return $query->count();
    }

    /**
     * @param string $related
     * @param string $idSubmission
     * @return array
     */
    public function removeSubmission(string $related, string $idSubmission)
    {
        if (FormUsers::isAdmin($related)) { //check if user is admin
            $submission = FormSubmission::find((int) $idSubmission);
            if ($submission) {
                event(new FormSubmissionDeletedEvent($submission));
                $submission->delete();
            }
        }

        return ['data' => null];
    }

    public function getFormAnswersUrl(Forms $form, bool $admin = false): ?string
    {
        $endpoint = match (true) {
            $form->form_type === 'surveys' => 'trends',
            $admin => 'admin-responses',
            default => 'my-responses'
        };

        return "forms/$endpoint/$form->uuid";
    }
}
