<?php

namespace Inside\Form\Services;

use Illuminate\Support\Facades\Auth;
use Inside\Content\Models\Contents\Forms;
use Inside\Content\Models\Contents\Users;
use Inside\Form\Facades\Form;
use Inside\Form\Facades\FormUsers;
use Inside\Form\Facades\SurveyAnswers;
use Inside\Form\Models\FormSubmission;

class FormAnswersService
{
    /**
     * @param mixed $fields
     * @param bool $export
     * @return array
     */
    private function getFormattedResults($fields, bool $export): array
    {
        $formattedResults = [];
        foreach ($fields as $field) {
            $formattedResults[] = [
                'field_id'           => $field->field_id,
                'field_order'        => $field->field_order,
                'field_question'     => $field->field_question,
                'field_choices'      => $field->field_choices,
                'field_type'         => $field->field_type,
                'field_closed_ended' => $field->field_closed_ended,
                'field_user_can'     => $export || ($field->show_in_list_user ?? false) || ($field->field_user_can->show_in_list_user ?? false),
                'field_reference'    => isset($field->field_select_type) && $field->field_select_type === 'content' ? $field->field_content_type : null,
            ];
        }

        return $formattedResults;
    }

    /**
     * @param FormSubmission $submission
     * @param Forms $content
     * @param bool $admin
     * @return array
     */
    private function setBaseResponse(FormSubmission $submission, Forms $content, bool $admin): array
    {
        $response = [
            'common' => [
                'uuid'         => $submission->id,
                'request_date' => $submission->created_at,
                'status'       => $submission->status,
                'code'         => $submission->code,
            ],
            'fields' => [],
        ];

        if (! $content->anonym || ! $admin) {
            $userModel = Users::find($submission->user_uuid);

            if ($userModel === null) {
                return [];
            }

            $response['user'] = [
                'uuid'      => $userModel->uuid,
                'firstname' => $userModel->firstname,
                'lastname'  => $this->formatUserFieldResult($userModel, 'lastname'),
                'isAdmin'   => FormUsers::isAdmin($content->uuid),
            ];
        }

        return $response;
    }

    /**
     * @param Forms $content
     * @param FormSubmission $submission
     * @param mixed $object
     * @param array $answer
     * @return array
     */
    private function getAnswerResponse(Forms $content, FormSubmission $submission, $object, array $answer)
    {
        foreach ($submission->answers as $submissionAnswer) {
            if ($submissionAnswer->field_id === $object['field_id']) {
                // if we get a reference and the reference is set we get the content title
                if ($object['field_reference'] && $submissionAnswer->answer) {
                    $query = call_user_func(type_to_class($object['field_reference']).'::query');
                    $content = $query->find($submissionAnswer->answer);
                    $answer['response'] = $content->uuid;
                } else {
                    $answer['response'] = $submissionAnswer->answer;
                }
                if ($object['field_type'] === 'input-attachment') {
                    $answer = Form::prepareFileFieldAnswer($content, $answer);
                }
            }
        }

        return $answer;
    }

    private function formatUserFieldResult(Users $user, string $field): string
    {
        return empty(config("form.append.$field")) ? $user->$field ?? '' :
            sprintf('%s (%s)', $user->$field ?? '', $user->{config("form.append.$field")} ?? '');
    }

    /**
     * @param Forms $content
     * @param bool $trending
     * @param bool $admin
     * @param bool $export
     * @return array|null
     */
    public function getFormAnswers(Forms $content, bool $trending = false, bool $admin = false, bool $export = false, bool $paginate = false, int $paginateLimit = 0): ?array
    {
        if ($trending) {
            return SurveyAnswers::getSurveyAnswers($content); //Use for form trending page
        }
        $fields = json_decode($content->inputs);
        $user = $admin ? null : Auth::user()->uuid;
        $formattedResults = $this->getFormattedResults($fields, $export);

        $submissions = FormSubmission::where('answerable_uuid', $content->uuid)->orderBy('created_at', 'DESC');
        if (! $admin) {
            $submissions = $submissions->where('user_uuid', $user);
        }

        $submissions = $paginate ? $submissions->paginate($paginateLimit) : $submissions->get();

        $responses = [];
        foreach ($submissions as $submission) {
            $response = $this->setBaseResponse($submission, $content, $admin);
            if (empty($response)) {
                continue;
            }

            foreach ($formattedResults as $object) {
                if (! $object['field_user_can']) {
                    continue;
                }
                $answer = [
                    'field_id' => $object['field_id'],
                    'field_type' => $object['field_type'],
                    'response' => null,
                ];

                $answer = $this->getAnswerResponse($content, $submission, $object, $answer);

                $response['fields'][] = $answer;
            }

            $responses[] = $response;
        }

        if (! $paginate) {
            return [
                'results' => $responses,
            ];
        }

        return [
            'results' => $responses,
            'current_page' => $submissions->currentPage(),
            'last_page' => $submissions->lastpage(),
            'per_page' => $submissions->perPage(),
        ];
    }
}
