<?php

namespace Inside\Form\Notifications;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Facades\Lang;
use Inside\Authentication\Models\User;
use Inside\Form\Models\FormSubmission;
use Inside\Notify\Messages\MailMessage;
use Inside\Notify\Notifications\MailNotification;

class FormSubmissionSummaryMailNotification extends MailNotification implements ShouldQueue
{
    public function toMail(mixed $notifiable): MailMessage
    {
        if (! $this->model instanceof FormSubmission) {
            return parent::toMail($notifiable);
        }

        $mail = $this->prepareMail();

        /** @var User $author */
        $author = User::find($this->model->user_uuid);
        $date = date('d/m/Y', $this->model->edited_at ?? $this->model->created_at);

        $comment = $this->model->comment
            ? Lang::get('notifications.form.submission.email.summary.comment', ['comment' => $this->model->comment], $this->user?->langcode)
            : null;

        $summaryTitle = Lang::get('notifications.form.submission.email.summary.title');

        /** @var User $author */
        $author = User::withoutGlobalScopes()->findOrFail($this->model->user_uuid);
        $submission = json_decode($this->model->answerable_model->inputs, true);

        $textFields = array_filter($submission, function ($field) {
            return in_array($field['field_type'], ['input-text', 'input-textarea']);
        });

        $answers = $this->model->answers;

        if ($this->model->answerable_model->anonym) {
            $summary = [
                Lang::get('notifications.form.submission.email.summary.date', ['date' => $date], $this->user?->langcode),
                Lang::get('notifications.form.submission.email.summary.anonym', [], $this->user?->langcode),
            ];
        } else {
            $summary = [
                Lang::get('notifications.form.submission.email.summary.name', ['firstname' => $author->firstname, 'lastname' => $author->lastname], $this->user?->langcode),
                Lang::get('notifications.form.submission.email.summary.email', ['email' => $author->email], $this->user?->langcode),
                Lang::get('notifications.form.submission.email.summary.date', ['date' => $date], $this->user?->langcode),
            ];
        }

        $summaryAnswers = [];

        foreach ($textFields as $field) {
            foreach ($answers as $answer) {
                if ($field['field_id'] === $answer['field_id']) {
                    if (empty($answer['answer'])) {
                        $summaryAnswers[] = $field['field_question'].': '.Lang::get('notifications.form.submission.email.summary.no.answer', [], $this->user?->langcode);
                        continue;
                    }
                    $summaryAnswers[] = $field['field_question'].': '.$answer['answer'];
                }
            }
        }

        if ($comment) {
            $mail['contents'][] = ''; // add blank line
            $mail['contents'][] = "$comment";
        }

        $mail['contents'][] = ''; // add blank line
        $mail['contents'][] = "$summaryTitle :";
        foreach ($summary as $summaryItem) {
            $mail['contents'][] = "- $summaryItem";
        }

        if (! empty($summaryAnswers)) {
            $mail['contents'][] = ''; // add blank line
            $mail['contents'][] = Lang::get('notifications.form.submission.email.summary.answers', [], $this->user?->langcode).' :';
            foreach ($summaryAnswers as $summaryAnswer) {
                $mail['contents'][] = "- $summaryAnswer";
            }
        }

        return $this->createMailMessage($mail);
    }
}
