<?php

namespace Inside\Form\Notifications;

use Illuminate\Contracts\Queue\ShouldQueue;
use Inside\Form\Models\FormAnswer;
use Inside\Form\Models\FormSubmission;
use Inside\Notify\Messages\MailMessage;
use Inside\Notify\Notifications\MailNotification;

class FormMailNotification extends MailNotification implements ShouldQueue
{
    protected function getValueFromAnswer(mixed $field): null | string | array
    {
        if (! $this->model instanceof FormSubmission) {
            return null;
        }

        $value = null;

        /** @var FormAnswer $answer */
        foreach ($this->model->answers as $answer) {
            if ($answer->field_id !== $field->field_id) {
                continue;
            }

            $value = match ($field->field_type) {
                'date' => date('d/m/Y', strtotime(strval($answer->answer))),
                default => $answer->answer,
            };
        }

        return $value;
    }

    /**
     * @param mixed $notifiable
     * @return MailMessage
     *
     * This argument is here because this function is an override
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function toMail($notifiable): MailMessage
    {
        $mail = $this->prepareMail();
        if (! $this->model instanceof FormSubmission) {
            return $this->createMailMessage($mail);
        }

        $fields = json_decode($this->model->answerable_model->inputs);

        foreach ($fields as $field) {
            if (! isset($field->field_mail) || ! $field->field_mail) {
                continue;
            }

            $label = $field->field_label;
            $value = $this->getValueFromAnswer($field);

            if (! empty($value) && ! empty($label)) {
                if (is_array($value)) {
                    $value = '['.implode(', ', $value).']';
                }
                $mail['contents'][] = "\n - **$label**: *$value*";
            }
        }

        return $this->createMailMessage($mail);
    }
}
