<?php

namespace Inside\Form\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Notification;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\Forms;
use Inside\Form\Events\FormSubmissionStatusEditedEvent;
use Inside\Form\Facades\Form;
use Inside\Form\Models\FormSubmission;
use Inside\Form\Notifications\FormSubmissionSummaryMailNotification;
use Inside\Notify\Listeners\BaseNotificationListener;
use Inside\Notify\Models\NotificationType;

class FormStatusListener extends BaseNotificationListener implements ShouldQueue
{
    /**
     * @param FormSubmissionStatusEditedEvent $event
     * @return void
     */
    public function handle(FormSubmissionStatusEditedEvent $event): void
    {
        $model = $event->submission;
        $userValidatesForm = $event->user;

        if ($model->notify) {
            $this->notifyAuthor($model, $userValidatesForm?->uuid);
        }

        if (config('notify_answer_to_admins')) {
            $this->notifyAdmins($model, $userValidatesForm?->uuid);
        }
    }

    private function notifyAdmins(FormSubmission $model, ?string $validator): void
    {
        $form = $model->answerable_model;

        if (empty($form->admin_roles)) {
            return;
        }

        $roles = json_decode($form->admin_roles);
        $users = DB::table('inside_users_roles')->whereIn('role_id', $roles)->get();

        $types = NotificationType::where(
            [
                'event' => 'Inside\Form\Events\FormSubmissionStatusEditedEvent',
                'action' => 'statusUpdatedAdmin',
            ]
        )->get();

        if (! config('form.notification_email_status')) {
            $types = $types->where('via', 'web');
        }

        $url = Form::getFormAnswersUrl(form:$model->answerable_model, admin:true);

        /** @var NotificationType $type */
        foreach ($types as $type) {
            foreach ($users as $user) {
                if ($user->user_uuid === $validator) {
                    continue;
                }

                /** @var User $user */
                $user = User::find($user->user_uuid);

                $this->when = $this->getInterval($type, $model);
                $this->route = $user;

                if (! $this->checkConditions($type, $model, $user)) {
                    continue;
                }

                $this->notify(
                    $type,
                    $user,
                    $model,
                    [
                        'from' => $model->user_uuid,
                        'form_title' => $model->answerable_model->title,
                        'form_id' => $model->code,
                        'url' => $url,
                    ]
                );
            }
        }
    }

    private function notifyAuthor(FormSubmission $model, ?string $validator): void
    {
        $types = NotificationType::where(
            [
                'event' => 'Inside\Form\Events\FormSubmissionStatusEditedEvent',
                'action' => 'statusUpdated',
            ]
        )->get();

        if (! config('form.notification_email_status')) {
            $types = $types->where('via', 'web');
        }

        $url = Form::getFormAnswersUrl($model->answerable_model);

        /** @var User $author */
        $author = User::find($model->user_uuid);
        foreach ($types as $type) {
            /** @var NotificationType $type */
            $this->when = $this->getInterval($type, $model);
            $this->route = $author;
            if (! $this->checkConditions($type, $model, $author)) {
                continue;
            }
            $this->notify(
                $type,
                $author,
                $model,
                [
                    'from' => $validator,
                    'form_title' => $model->answerable_model->title,
                    'form_id' => (string) $model->id,
                    'url' => $url,
                ]
            );
        }
    }

    public function notify(NotificationType $type, ?User $user, mixed $model = null, array $data = [], array $roles = [], ?string $theme = null): void
    {
        if ($type->via === 'email') { // If email notification, we generate FormStatusUpdatedMailNotification and send it.
            $notification = new FormSubmissionSummaryMailNotification($type, $user, $model, $data);
            Notification::route('mail', $user?->email);
            $this->sendNotification($notification, $model);
        } else { // Else, we use the classic way.
            parent::notify($type, $user, $model, $data, $roles, $theme);
        }
    }
}
