<?php

namespace Inside\Form\Services;

use Inside\Content\Models\Contents\Forms;
use Inside\Form\Models\FormAnswer;

class SurveyAnswerService
{
    /**
     * @param array<string, mixed> $answers
     * @param array<int, array<string, mixed>> $formattedResults
     * @return array<string, mixed>
     */
    private function getAnswersByField(array $answers, array $formattedResults): array
    {
        $answersByField = [];
        $keys = array_keys($answers);
        foreach ($keys as $index) {
            foreach ($formattedResults as $result) {
                if ($result['field_id'] == $index) {
                    $answersByField[$index] = $answers[$index];
                }
            }
        }

        return $answersByField;
    }

    /**
     * @param string $fieldType
     * @param array<mixed> $answersValues
     * @return array<mixed>|float|int|null
     */
    private function getResponseFieldIndex(string $fieldType, array $answersValues)
    {
        if ($fieldType === 'input-checkbox') {
            return array_count_values(call_user_func_array('array_merge', $answersValues));
        } else {
            if ($fieldType === 'input-radio' || $fieldType === 'input-selectbox') {
                return array_count_values(array_filter($answersValues));
            } else {
                if ($fieldType === 'input-rating') {
                    return array_sum(array_filter($answersValues)) / count(array_filter($answersValues));
                }

                return null;
            }
        }
    }

    /**
     * @param array<string, mixed> $answers
     * @param array<int, array<string, mixed>> $formattedResults
     * @return array<int|string,array<mixed>|float|int|null>
     */
    private function getSurveyResponses(array $answers, array $formattedResults): array
    {
        $answersByField = $this->getAnswersByField($answers, $formattedResults);
        $keys = array_keys($answers);
        foreach ($keys as $index) {
            foreach ($formattedResults as $result) {
                if ($result['field_id'] == $index) {
                    $answersByField[$index] = $answers[$index];
                }
            }
        }
        $responses = [];
        foreach ($answersByField as $fieldId => $answers) {
            $answersValues = array_column($answers, 'answer');
            $fieldIndex = 0;
            foreach ($formattedResults as $index => $object) {
                if ($fieldId == $object['field_id']) {
                    $fieldIndex = $index;
                    break;
                }
            }

            $responses[$fieldIndex] = null;
            $fieldType = $formattedResults[$fieldIndex]['field_type'];
            $responses[$fieldIndex] = $this->getResponseFieldIndex($fieldType, $answersValues);
        }

        return $responses;
    }

    /**
     * @param mixed $fields
     * @return array<int,array<string, mixed>>
     */
    private function getFormattedResults($fields): array
    {
        $formattedResults = [];
        foreach ($fields as $field) {
            if (! $field->field_closed_ended) {
                continue;
            }

            $formattedResults[] = [
                'field_id'           => $field->field_id,
                'field_order'        => $field->field_order,
                'field_question'     => $field->field_question,
                'field_choices'      => $field->field_choices,
                'field_type'         => $field->field_type,
                'field_closed_ended' => $field->field_closed_ended,
            ];
        }

        return $formattedResults;
    }

    /**
     * @param Forms $content
     * @return array|mixed[][]|null
     */
    public function getSurveyAnswers(Forms $content): ?array
    {
        $fields = json_decode($content->inputs);
        $formattedResults = $this->getFormattedResults($fields);

        $answers = FormAnswer::whereHas(
            'submission',
            function ($q) use ($content) {
                $q->where('answerable_uuid', $content->uuid);
            }
        )->get()->groupBy('field_id')->toArray();

        $responses = $this->getSurveyResponses($answers, $formattedResults);

        foreach ($formattedResults as $index => $object) {
            if ($object['field_type'] === 'input-checkbox' || $object['field_type'] === 'input-radio' || $object['field_type'] === 'input-selectbox') {
                $formattedResults[$index]['field_responses'] = [];

                foreach ($object['field_choices'] as $choice) {
                    if (! array_key_exists($index, $responses)) {
                        continue;
                    }
                    $responses[$index] = (array) $responses[$index];
                    // if a choice has never been used as answer
                    if (! array_key_exists($choice, $responses[$index])) {
                        continue;
                    }
                    $value = $responses[$index][$choice];
                    $formattedResults[$index]['field_responses'][] = [
                        'label' => $choice,
                        'value' => $value ?? 0,
                    ];
                }
            } else {
                if ($object['field_type'] === 'input-rating') {
                    $formattedResults[$index]['field_responses'][] = [
                        'label' => $object['field_question'],
                        'value' => $responses[$index],
                    ];
                }
            }
        }

        return [
            'results' => $formattedResults,
        ];
    }
}
