<?php

namespace Inside\Form\Services;

use Carbon\Carbon;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Inside\Content\Exceptions\ValidatorException;
use Inside\User\Models\User;
use Symfony\Component\HttpFoundation\File\MimeType\ExtensionGuesser;
use Symfony\Component\HttpFoundation\File\MimeType\MimeTypeGuesser;

class FormFileService
{
    /**
     * @param string $path
     * @return string
     */
    private function getExtension(string $path)
    {
        $mime = $this->getMime($path);
        $guesser = ExtensionGuesser::getInstance();

        return $guesser->guess($mime);
    }

    /**
     * @param string $path
     * @return string
     */
    private function getMime(string $path)
    {
        $guesser = MimeTypeGuesser::getInstance();

        return $guesser->guess($path);
    }

    /**
     * @param int $bytes
     * @return string
     */
    private function bytesToHuman(int $bytes)
    {
        /** @var User $me */
        $me = Auth::user();
        $locale = $me->langcode ?? config('app.locale');
        if ($locale == 'fr') {
            $units = ['o', 'Ko', 'Mo', 'Go', 'To', 'Po'];
        } else {
            $units = ['B', 'KiB', 'MiB', 'GiB', 'TiB', 'PiB'];
        }

        for ($i = 0; $bytes > 1024; $i++) {
            $bytes /= 1024;
        }

        return round($bytes, 2).' '.$units[$i];
    }

    /**
     * Save data file from upload data ( data: style ) to storage
     * and return informations on saved file
     *
     * @param string $formUuid
     * @param string $temporaryPath
     * @return array
     */
    public function saveFile(int $formSubmissionId, string $formUuid, string $temporaryPath): array
    {
        $disk = Storage::disk('local');

        // Reset Mime guesser,drupal mess everything
        MimeTypeGuesser::reset();

        $allowedFileTypes =
            explode(',', 'doc,docx,odt,txt,pdf,xls,xlsx,csv,ppt,pptx,ai,jpeg,jpg,gif,png,svg,tif,mp4,avi,mov,webm,dwg,rtf');

        // Build validator
        $validator = Validator::make(
            ['temp' => $temporaryPath],
            [
                'temp' => [
                    'required',
                    // Supported mime
                    function ($attribute, $value, $fail) use ($allowedFileTypes, $disk) {
                        $extension = $this->getExtension($disk->path($value));
                        if (! in_array($extension, $allowedFileTypes)) {
                            $fail(
                                Lang::get(
                                    'validation.file.wrongextension',
                                    ['attribute' => $attribute, 'type' => $extension]
                                )
                            );
                        }
                    },
                    // Max size
                    function ($attribute, $value, $fail) use ($disk) {
                        $maxSize = UploadedFile::getMaxFilesize();
                        if (File::size($disk->path($value)) > $maxSize) {
                            $fail(
                                Lang::get(
                                    'validation.file.maxsized',
                                    ['attribute' => $attribute, 'maxsize' => $this->bytesToHuman((int) $maxSize)]
                                )
                            );
                        }
                    },
                ],
            ]
        );
        if ($validator->fails()) {
            throw new ValidatorException($validator);
        }

        // Validation succeed
        $fileName = md5(Carbon::now()->format('Y-m-d-his').Str::random(40)).'.'.pathinfo($temporaryPath, PATHINFO_EXTENSION);
        $file = 'forms/'.$formUuid.'/'.$formSubmissionId.'-'.$fileName;
        if (Storage::move($temporaryPath, $file)) {
            return [
                'file' => $file,
                'name' => pathinfo($temporaryPath, PATHINFO_FILENAME),
            ];
        }

        return [];
    }

    public function deleteFile(string $file): void
    {
        Storage::disk('local')->delete($file);
    }
}
