<?php

namespace Inside\Form\Services;

use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Storage;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\Forms;
use Inside\Content\Models\Contents\Users;
use Inside\Form\Facades\Form;
use Inside\Form\Facades\FormUsers;
use Inside\Form\Models\FormAnswer;
use League\Csv\Writer;

class FormExporterService
{
    /**
     * @param string $related
     * @return array
     */
    public function exportSubmissions(string $related): array
    {
        $form = Forms::where('inside_content_forms.uuid', $related)->first();

        if ($form && $form->form_type == 'surveys') {
            return $this->exportSurvey($form);
        } else {
            return $this->exportForm($form);
        }
    }

    /**
     * @param array $headers
     * @param FormAnswer $answer
     * @param array $choices
     * @return array
     *
     * @SuppressWarnings(PHPMD.CountInLoopExpression)
     */
    private function getSurveyExportUserAnswer(array $headers, $answer, array $choices): array
    {
        $line = [];
        // we don't care about elements 0 to 3 because we already have the infos we need
        // we just need from idx 4 because it's the content answers
        for ($idx = 4; $idx < count($headers); $idx++) {
            // this is our answer
            if (trim($headers[$idx]) === $answer->answer || is_array($answer->answer) && in_array($headers[$idx], $answer->answer)) {
                $line[] = '1';
            } elseif (in_array($headers[$idx], $choices)) {
                // this is not our answer but it's still a choice from this question
                $line[] = '0';
            } elseif ($answer->field_type === 'input-rating' && $headers[$idx] === 'note') {
                $line[] = $answer->answer;
            } else {
                // response from another question
                $line[] = '/';
            }
        }

        return $line;
    }

    /**
     * @param array|null $answers
     * @return array
     */
    private function getSurveyHeaders(?array $answers): array
    {
        $headers = ['questions', 'prénom', 'nom', 'adresse mail'];

        if (empty($answers)) {
            return $headers;
        }

        foreach ($answers['results'] as $answer) {
            if ($answer['field_type'] === 'input-rating') {
                $headers[] = 'note';
            } else {
                $headers = array_merge($headers, $answer['field_choices']);
            }
        }

        return $headers;
    }

    /**
     * @param mixed $inputs
     * @return string[]
     */
    private function getFormHeaders($inputs): array
    {
        $headers = ['Users ', Lang::getFromJSON('forms.id'), Lang::getFromJSON('forms.status'), Lang::getFromJSON('forms.date')]; //HACK

        foreach ($inputs as $input) {
            // we don't need to export the files attached
            if (
                $input->field_type === 'input-attachment' &&
                config('form.export_attachment_disabled')
            ) {
                continue;
            }
            $headers[] = $input->field_listing_name ?? $input->field_label ?? $input->field_question;
        }

        return $headers;
    }

    /**
     * @param array $headers
     * @return Writer
     * @throws \League\Csv\CannotInsertRecord
     * @throws \League\Csv\Exception
     */
    private function initCsv(array $headers): Writer
    {
        if (! Storage::disk('protected')->exists('exports')) {
            Storage::disk('protected')->makeDirectory('exports', 0755, true); //creates directory
        }
        $csv = Writer::createFromString();
        $csv->setDelimiter(';');
        $csv->setOutputBOM(Writer::BOM_UTF8);
        $csv->insertOne($headers);
        $csv->setEscape('');

        return $csv;
    }

    public function exportSurvey(Forms $form, string $userUuid = null): array
    {
        $authUser = User::find($userUuid);
        if (! FormUsers::isAdmin($form->uuid, $authUser)) {
            return [];
        }

        $answers = Form::getAnswers($form->uuid, false, false, false, false, false, 0, $userUuid);

        $headers = $this->getSurveyHeaders($answers);

        $title = str_slug($form->title).'-'.date('Y-m-d');
        $csv = $this->initCsv($headers);

        $inputs = json_decode($form->inputs);

        foreach ($inputs as $input) {
            $formAnswers = FormAnswer::whereHas(
                'submission',
                function ($q) use ($form) {
                    $q->where('answerable_uuid', $form->uuid);
                }
            )->where('field_id', $input->field_id)->get();
            foreach ($formAnswers as $answer) {
                $line = [];
                // find submission user and add his infos
                $uuid = $answer->submission->user_uuid;
                $user = Users::find($uuid);
                // tricks here because field_type is not defined
                $answer->field_type = collect($answers['results'])->where('field_id', $answer->field_id)->pluck('field_type')->first();
                $firstname = $form->anonym ? Lang::getFromJSON('anonymous.user.firstname') : $user->firstname;
                $lastname = $form->anonym ? Lang::getFromJSON('anonymous.user.lastname') : $user->lastname;
                $email = $form->anonym ? trim(Lang::getFromJSON('anonymous.user.firstname').' '.
                    Lang::getFromJSON('anonymous.user.lastname')) : $user->email;
                $line = array_merge(
                    [$input->field_question, $firstname, $lastname, $email],
                    $this->getSurveyExportUserAnswer($headers, $answer, $input->field_choices)
                );

                $csv->insertOne($line);
            }
        }

        Storage::disk('protected')->put('exports/'.$title.'.csv', $csv->getContent());

        $file = Storage::disk('protected')->exists('exports/'.$title.'.csv') ? Storage::disk('protected')->url('exports/'.$title.'.csv') : null;

        return [
            'url' => $file,
            'path' => 'exports/'.$title.'.csv',
        ];
    }

    /**
     * @param mixed $answer
     * @return array
     */
    private function getFormLine($answer, array $inputs): array
    {
        $line = [
            ($answer['user']['firstname'] ?? '').' '.($answer['user']['lastname'] ?? ''),
            $answer['common']['uuid'],
            Lang::getFromJSON('forms.status.'.$answer['common']['status']),
            get_date_in_user_timezone($answer['common']['request_date'], 'd/m/Y H:i'),
        ];

        $fields = $this->getSortedAnswerFields($answer['fields'], $inputs);

        foreach ($fields as $field) {
            if ($field['field_type'] === 'input-attachment' &&
                config('form.export_attachment_disabled')
            ) {
                continue;
            }
            $line[] = match ($field['field_type']) {
                'input-date' => is_null($field['response']) ? '' : get_date_in_user_timezone($field['response'], 'd/m/Y H:i'),
                'input-attachment' => $field['response'] ?? '',
                default => is_array($field['response']) ? implode(',', $field['response']) : $field['response'],
            };
        }

        return $line;
    }

    private function getSortedAnswerFields(array $fields, array $inputs): array
    {
        $fieldIds = collect($inputs)->sortBy('field_order')->pluck('field_id');

        return collect($fields)->sortBy(fn ($field) => $fieldIds->search($field['field_id']))->all();
    }

    private function getSortedInputs(Forms $form): array
    {
        return collect(json_decode($form->inputs))->sortBy('field_order')->all();
    }

    /**
     * @param Forms $form
     * @return array<string, string|null>
     * @throws \League\Csv\CannotInsertRecord
     * @throws \League\Csv\Exception
     */
    public function exportForm(Forms $form): array
    {
        if (! FormUsers::isAdmin($form->uuid)) {
            return [];
        }

        $answers = Form::getAnswers($form->uuid, false, false, false, true);
        $inputs = $this->getSortedInputs($form);
        $headers = $this->getFormHeaders($inputs);
        $title = str_slug($form->title).'-'.date('Y-m-d');
        $csv = $this->initCsv($headers);

        foreach ($answers['results'] as $answer) {
            $line = $this->getFormLine($answer, $inputs);
            $csv->insertOne($line);
        }

        Storage::disk('protected')->put('exports/'.$title.'.csv', $csv->getContent());

        $file = Storage::disk('protected')->exists('exports/'.$title.'.csv') ? Storage::disk('protected')->url('exports/'.$title.'.csv') : null;

        return [
            'url' => $file,
            'path' => 'exports/'.$title.'.csv',
        ];
    }
}
