<?php

namespace Inside\Form\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Date;
use Illuminate\Support\Facades\DB;
use Inside\Content\Models\Content;
use Inside\Content\Models\Contents\Forms;

/**
 * @method static Builder select($columns = ['*'])
 * @method static Builder where($column, $operator = null, $value = null, $boolean = 'and')
 * @method static static create(array $attributes = [])
 * @method static null|static find($id, $columns = ['*'])
 *
 * @property int $id
 * @property string $user_uuid
 * @property string|null $editor_uuid
 * @property int $created_at
 * @property int $updated_at
 * @property int|null $edited_at
 * @property int $status
 * @property Collection $answers
 * @property mixed $notify
 * @property string $answerable_type
 * @property string $answerable_uuid
 * @property string|null $comment
 * @property Forms $answerable_model
 * @property string $code
 */
class FormSubmission extends Content
{
    public const STATUS_REJECTED = 0;

    public const STATUS_VALIDATED = 1;

    public const STATUS_PENDING = 2;

    public const DEFAULT_STATUS = self::STATUS_PENDING;

    /**
     * The table associated with the model.
     */
    protected $table = 'inside_forms_submissions';

    /**
     * Casts the following attributes
     */
    protected $casts = [
        'user_uuid' => 'string',
        'editor_uuid' => 'string',
        'code' => 'string',
        'comment' => 'string',
        'created_at' => 'timestamp',
        'updated_at' => 'timestamp',
        'edited_at' => 'timestamp',
        'status' => 'integer',
        'notify' => 'integer',
    ];

    /**
     * The attributes that are mass assignable.
     */
    protected $fillable = [
        'user_uuid',
        'editor_uuid',
        'answerable_uuid',
        'answerable_type',
        'created_at',
        'edited_at',
        'comment',
        'status',
        'code',
        'notify',
    ];

    protected $attributes = [
        'status' => self::DEFAULT_STATUS,
    ];

    /**
     * The user that owns the answer.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo('Inside\Content\Models\Contents\Users', 'user_uuid', 'uuid');
    }

    public function getAnswerableModelAttribute(): Content
    {
        return call_user_func($this->answerable_type.'::find', $this->answerable_uuid);
    }

    public function answerable(): MorphTo
    {
        return $this->morphTo('answerable', 'answerable_type', 'answerable_uuid');
    }

    /**
     * Get the answers for the submission.
     */
    public function answers(): HasMany
    {
        return $this->hasMany('Inside\Form\Models\FormAnswer', 'submission_id', 'id');
    }

    public function getCodeAttribute(): string
    {
        return $this->attributes['code'] ?? (string) $this->getKey();
    }

    /**
     * Magic isset an answer
     */
    public function __isset($key)
    {
        $form = Forms::find($this->getAttribute('answerable_uuid'));
        if ($form) {
            $inputs = collect(json_decode($form->inputs));
            $field = $inputs->where('field_slug', $key)->first();
            if ($field) {
                $answer = DB::table('inside_forms_answers')->where('field_id', $field->field_id)->where('submission_id', $this->id)
                    ->first();
                if ($answer) {
                    return true;
                }
            }
        }

        return parent::__isset($key);
    }

    protected function setAuthorAttribute(mixed $value): void
    {
    }

    protected function getAuthorAttribute(): string
    {
        return $this->user_uuid;
    }

    protected function setAuthorIdAttribute(mixed $value): void
    {
    }

    protected function getAuthorIdAttribute(): string
    {
        return $this->user_uuid;
    }

    protected function setEditedAtAttribute(mixed $value): void
    {
        if (is_int($value)) {
            $value = Date::createFromTimestamp($value);
        }

        $this->attributes['edited_at'] = $value;
    }

    /**
     * Magic get an answer
     */
    public function __get($key)
    {
        $form = Forms::find($this->getAttribute('answerable_uuid'));
        if ($form) {
            $inputs = collect(json_decode($form->inputs));
            $field = $inputs->where('field_slug', $key)->first();
            if ($field) {
                $answer = DB::table('inside_forms_answers')->where('field_id', $field->field_id)->where('submission_id', $this->id)
                    ->first();
                if ($answer) {
                    return json_decode($answer->answer);
                }
            }
        }

        return parent::__get($key);
    }
}
