<?php

declare(strict_types=1);

namespace Inside\Form\Listeners;

use Exception;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\Forms;
use Inside\Form\Events\FormSubmitedEvent;
use Inside\Form\Facades\Form;
use Inside\Form\Models\FormSubmission;
use Inside\Form\Notifications\FormMailNotification;
use Inside\Notify\Listeners\BaseNotificationListener;
use Inside\Notify\Models\NotificationType;

final class FormSubmittedUsersListener extends BaseNotificationListener implements ShouldQueue
{
    protected function notifyUsers(NotificationType $type, Collection $users, Forms $form, FormSubmission $submission): void
    {
        $fromUser = $submission->user()->first();
        $fromFullName = Str::ucfirst(Str::lower($fromUser->firstname)).' '.Str::upper($fromUser->lastname);
        $notificationData = [
            'from' => $form->anonym === false ? $submission->user_uuid : null,
            'from_full_name' => $form->anonym === false ? $fromFullName : null,
            'url' => Form::getFormAnswersUrl(form: $submission->answerable_model, admin: true),
            'form_title' => $form->title,
            'form_id' => $submission->code,
        ];

        $users->each(function (User $user) use ($submission, $type, $notificationData) {
            if ($user->uuid === $submission->user_uuid) {
                return;
            }

            $this->when = $this->getInterval($type, $submission);
            $this->route = $user;

            if (! $this->checkConditions($type, $submission, $user)) {
                return;
            }

            try {
                $this->notify(
                    $type,
                    $user,
                    $submission,
                    $notificationData,
                );
            } catch (Exception $exception) {
                Log::info($exception->getMessage());
            }
        });
    }

    public function handle(FormSubmitedEvent $event): void
    {
        $model = $event->submission;
        $form = $model->answerable_model;

        if (empty($form->admin_roles)) {
            return;
        }

        $roles = json_decode($form->admin_roles);

        if (! is_array($roles) || empty($roles)) {
            return;
        }

        $users = User::whereHas('roles', fn (Builder $query) => $query->whereIn('role_id', $roles))->get();

        /** @var Collection<NotificationType> $types */
        $types = NotificationType::where(
            [
                ['event', 'Inside\Form\Events\FormSubmitedEvent'],
                ['action', 'submit'],
                ['model', 'Inside\Form\Models\FormSubmission'],
            ]
        )->get();

        foreach ($types as $type) {
            $this->notifyUsers($type, $users, $form, $model);
        }
    }

    /**
     * @param NotificationType $type
     * @param mixed $user
     * @param mixed $model
     * @param array $data
     * @param mixed $roles
     * @param string|null $theme
     * @return void
     * @throws Exception
     */
    public function notify(NotificationType $type, $user, $model = null, array $data = [], $roles = [], ?string $theme = null): void
    {
        if ($type->via === 'email') { // If email notification, we generate FormMailNotification and send it.
            $notification = new FormMailNotification($type, $user, $model, $data);
            Notification::route('mail', $user->email);
            $this->sendNotification($notification, $model);
        } else { // Else, we use the classic way.
            parent::notify($type, $user, $model, $data, $roles, $theme);
        }
    }
}
