<?php

namespace Inside\Form\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inside\Authentication\Models\User;
use Inside\Form\Facades\Form;
use Inside\Form\Facades\FormUsers;
use Inside\Form\Jobs\ExportSubmissionsJob;
use Inside\Form\Models\FormSubmission;
use Inside\Support\Str;

class FormController
{
    public function storeAnswers(Request $request): JsonResponse
    {
        $answers = $request->get('answers', []);
        $data = $request->except('answers');
        $data['answerable_uuid'] = $data['related_uuid'];
        $data['answerable_type'] = 'Inside\Content\Models\Contents\\'.Str::studly($data['related_type']);

        unset($data['related_uuid']);
        unset($data['related_type']);

        Form::storeAnswers($data, $answers);

        return response()->json([], 201);
    }

    public function updateSubmission(Request $request, string $related, int $submission): JsonResponse
    {
        /** @var FormSubmission $formSubmission */
        $formSubmission = FormSubmission::where('id', $submission)
            ->where('answerable_uuid', $related)
            ->firstOrFail();

        /** @var User $currentUser */
        $currentUser = Auth::user();

        if (! $currentUser->isSuperAdmin() && $formSubmission->user_uuid !== $currentUser->uuid) {
            abort(403, 'Only its author can edit a submission');
        }

        $answers = $request->get('answers', []);

        Form::updateSubmissionAnswers($formSubmission, $answers, $currentUser->uuid);

        return response()->json(Form::getSubmission($related, $submission));
    }

    /**
     * @param string $related
     * @return array
     */
    public function getAnswers(Request $request, string $related): array
    {
        $forceSurvey = $request->get('force_survey', false);
        $trending = $request->get('trending', false);
        $myAnswers = $request->get('my_answers', 'false');

        $paginate = false;
        $paginationLimit = 0;
        if ($request->has('filters')) {
            $filters = json_decode($request->get('filters'), true) ?? [];
            $paginate = $filters['paginate'] ?? false;
            $paginationLimit = $filters['limit'] ?? 0;
        }

        return Form::getAnswers($related, $forceSurvey, $trending, json_decode($myAnswers), false, $paginate, $paginationLimit);
    }

    /**
     * @param string $related
     * @param int $submission
     * @return array
     */
    public function getSubmission(string $related, int $submission): array
    {
        return Form::getSubmission($related, $submission);
    }

    /**
     * @param int $submission
     * @return array
     */
    public function editSubmissionStatus(Request $request, int $submission)
    {
        $status = $request->get('status');
        $notify = $request->get('notify');
        $comment = $request->get('comment');

        return Form::editSubmissionStatus($status, $submission, $notify, $comment);
    }

    /**
     * @param string $related
     * @return int
     */
    public function getAnswersCount(string $related, string $user = null): int
    {
        return Form::getAnswersCount($related, $user);
    }

    /**
     * @param string $related
     * @return array
     */
    public function checkUserAdmin(string $related): array
    {
        return FormUsers::checkUserAdmin($related);
    }

    /**
     * @param string $related
     * @param string $submission
     * @return array
     */
    public function deleteSubmission(string $related, string $submission)
    {
        return Form::removeSubmission($related, $submission);
    }

    /**
     * @param string $related
     * @return array
     */
    public function getContributors(string $related)
    {
        return FormUsers::getContributors($related);
    }

    /**
     * @param string $related
     * @return JsonResponse
     */
    public function exportSubmissions(string $related): JsonResponse
    {
        ExportSubmissionsJob::dispatch($related, auth()->user()->uuid)
            ->onQueue(get_low_priority_queue_name());

        return response()->json(['message' => 'success']);
    }
}
