<?php

namespace Drupal\inside\Plugin\Field\FieldWidget;

use Drupal\Core\Field\Plugin\Field\FieldWidget\StringTextfieldWidget;
use Drupal\Core\Form\FormStateInterface;
use Egulias\EmailValidator\EmailValidator;
use Egulias\EmailValidator\Validation\DNSCheckValidation;
use Egulias\EmailValidator\Validation\MultipleValidationWithAnd;
use Egulias\EmailValidator\Validation\NoRFCWarningsValidation;
use Egulias\EmailValidator\Validation\RFCValidation;
use Egulias\EmailValidator\Validation\SpoofCheckValidation;
use Illuminate\Support\Str;

/**
 * Plugin implementation of the 'inside_link_field_widget' widget.
 *
 * @FieldWidget(
 *   id = "inside_link_field_widget",
 *   module = "inside",
 *   label = @Translation("Inside link field widget"),
 *   field_types = {
 *     "string"
 *   }
 * )
 */
class InsideLinkFieldWidget extends StringTextfieldWidget
{
    protected $allowedSchemes = [
        'http',
        'https',
        'file',
        'mailto',
    ];

    public static function defaultSettings()
    {
        return [
                'size'        => 2048,
                'placeholder' => 'https//example.com',
            ] + parent::defaultSettings();
    }

    public function settingsForm(array $form, FormStateInterface $form_state)
    {
        return [];
    }

    public function massageFormValues(array $values, array $form, FormStateInterface $form_state)
    {
        foreach ($values as &$item) {
            if (!$this->validateUrl($item['value'])) {
                $item['value'] = null;
            }
        }

        return $values;
    }

    /**
     * Validate an $url ( only certain schemes are allowed )
     *
     * @param string $url
     * @return bool
     */
    protected function validateUrl(string $url): bool
    {
        if (!is_string($url)) {
            return false;
        }

        foreach ($this->allowedSchemes as $scheme) {
            $method = 'validate' . ucfirst($scheme) . 'Scheme';
            if (method_exists($this, $method)) {
                if ($this->{$method}($url)) {
                    return true;
                }
            } elseif ($this->validateCommonSchemes($url, $scheme)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Validate a basic url
     *
     * @param string $url
     * @return bool
     */
    public function validateBasic(string $url): bool
    {
        return filter_var($url, FILTER_VALIDATE_URL) !== false;
    }

    /**
     * Validate with common schemes
     *
     * @param string $url
     * @param null   $scheme
     * @return bool
     */
    protected function validateCommonSchemes(string $url, $scheme = null): bool
    {
        if (!$scheme) {
            return $this->validateBasic($url) && (bool)preg_match("/^\w+:\/\//i", $url);
        } else {
            return $this->validateBasic($url) && (bool)preg_match("/^{$scheme}:\/\//", $url);
        }
    }

    /**
     * validate a mailto url
     *
     * @param $url
     * @return bool
     */
    public function validateMailtoScheme($url): bool
    {
        return $this->validateBasic($url) && preg_match("/^mailto:/", $url)
            && $this->validateEmail(
                Str::after($url, 'mailto:'),
                ['rfc', 'strict', 'dns']
            );
    }

    /**
     * Nicely validate an email
     *
     * @param string $email
     * @param array  $parameters
     * @return bool
     */
    protected function validateEmail(string $email, array $parameters = []): bool
    {
        if (!is_string($email)) {
            return false;
        }

        $validations = collect($parameters)->unique()->map(
            function ($validation) {
                if ($validation === 'rfc') {
                    return new RFCValidation;
                } elseif ($validation === 'strict') {
                    return new NoRFCWarningsValidation;
                } elseif ($validation === 'dns') {
                    return new DNSCheckValidation;
                } elseif ($validation === 'spoof') {
                    return new SpoofCheckValidation;
                }
            }
        )->values()->all() ?: [new RFCValidation];

        $validator = new EmailValidator;

        return (new EmailValidator)->isValid($email, new MultipleValidationWithAnd($validations));
    }

    /**
     * Validate an https url
     *
     * @param string $url
     * @return bool
     */
    public function validateHttpsScheme(string $url): bool
    {
        if (!is_string($url) || !$this->validateBasic($url)) {
            return false;
        }
        $pattern = '~^
            (http|https)://                                 # protocol
            (((?:[\_\.\pL\pN-]|%[0-9A-Fa-f]{2})+:)?((?:[\_\.\pL\pN-]|%[0-9A-Fa-f]{2})+)@)?  # basic auth
            (
                ([\pL\pN\pS\-\_\.])+(\.?([\pL\pN]|xn\-\-[\pL\pN-]+)+\.?) # a domain name
                    |                                                 # or
                \d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}                    # an IP address
                    |                                                 # or
                \[
                    (?:(?:(?:(?:(?:(?:(?:[0-9a-f]{1,4})):){6})(?:(?:(?:(?:(?:[0-9a-f]{1,4})):(?:(?:[0-9a-f]{1,4})))|(?:(?:(?:(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9]))\.){3}(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9])))))))|(?:(?:::(?:(?:(?:[0-9a-f]{1,4})):){5})(?:(?:(?:(?:(?:[0-9a-f]{1,4})):(?:(?:[0-9a-f]{1,4})))|(?:(?:(?:(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9]))\.){3}(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9])))))))|(?:(?:(?:(?:(?:[0-9a-f]{1,4})))?::(?:(?:(?:[0-9a-f]{1,4})):){4})(?:(?:(?:(?:(?:[0-9a-f]{1,4})):(?:(?:[0-9a-f]{1,4})))|(?:(?:(?:(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9]))\.){3}(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9])))))))|(?:(?:(?:(?:(?:(?:[0-9a-f]{1,4})):){0,1}(?:(?:[0-9a-f]{1,4})))?::(?:(?:(?:[0-9a-f]{1,4})):){3})(?:(?:(?:(?:(?:[0-9a-f]{1,4})):(?:(?:[0-9a-f]{1,4})))|(?:(?:(?:(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9]))\.){3}(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9])))))))|(?:(?:(?:(?:(?:(?:[0-9a-f]{1,4})):){0,2}(?:(?:[0-9a-f]{1,4})))?::(?:(?:(?:[0-9a-f]{1,4})):){2})(?:(?:(?:(?:(?:[0-9a-f]{1,4})):(?:(?:[0-9a-f]{1,4})))|(?:(?:(?:(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9]))\.){3}(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9])))))))|(?:(?:(?:(?:(?:(?:[0-9a-f]{1,4})):){0,3}(?:(?:[0-9a-f]{1,4})))?::(?:(?:[0-9a-f]{1,4})):)(?:(?:(?:(?:(?:[0-9a-f]{1,4})):(?:(?:[0-9a-f]{1,4})))|(?:(?:(?:(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9]))\.){3}(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9])))))))|(?:(?:(?:(?:(?:(?:[0-9a-f]{1,4})):){0,4}(?:(?:[0-9a-f]{1,4})))?::)(?:(?:(?:(?:(?:[0-9a-f]{1,4})):(?:(?:[0-9a-f]{1,4})))|(?:(?:(?:(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9]))\.){3}(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9])))))))|(?:(?:(?:(?:(?:(?:[0-9a-f]{1,4})):){0,5}(?:(?:[0-9a-f]{1,4})))?::)(?:(?:[0-9a-f]{1,4})))|(?:(?:(?:(?:(?:(?:[0-9a-f]{1,4})):){0,6}(?:(?:[0-9a-f]{1,4})))?::))))
                \]  # an IPv6 address
            )
            (:[0-9]+)?                              # a port (optional)
            (?:/ (?:[\pL\pN\-._\~!$&\'()*+,;=:@]|%[0-9A-Fa-f]{2})* )*          # a path
            (?:\? (?:[\pL\pN\-._\~!$&\'\[\]()*+,;=:@/?]|%[0-9A-Fa-f]{2})* )?   # a query (optional)
            (?:\# (?:[\pL\pN\-._\~!$&\'()*+,;=:@/?]|%[0-9A-Fa-f]{2})* )?       # a fragment (optional)
        $~ixu';

        return preg_match($pattern, $url) > 0;
    }

    /**
     * Validate an http url ( same as https )
     *
     * @param string $url
     * @return bool
     */
    public function validateHttpScheme(string $url): bool
    {
        return $this->validateHttpsScheme($url);
    }
}
