<?php

namespace Inside\Host\Normalizer;

use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Field\EntityReferenceFieldItemList;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\Core\Field\FieldItemList;
use Drupal\Core\Field\Plugin\Field\FieldType\LanguageItem;
use Drupal\Core\Field\Plugin\Field\FieldType\TimestampItem;
use Drupal\Core\TypedData\ListInterface;
use Drupal\file\Plugin\Field\FieldType\FileFieldItemList;
use Drupal\file\Plugin\Field\FieldType\FileItem;
use Drupal\link\Plugin\Field\FieldType\LinkItem;
use Drupal\serialization\EntityResolver\UuidReferenceInterface;
use Drupal\serialization\EntityResolver\UuidResolver;
use Drupal\serialization\Normalizer\NormalizerBase;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Str;
use Log;
use Symfony\Component\Serializer\Normalizer\DenormalizerInterface;

/**
 * List Normalizer
 *
 * @category  Class
 * @package   Inside\Host\Normalizer\ListNormalizer
 * @author    Maecia <technique@maecia.com>
 * @author    Nicolas Deniaud <contact@feldoe.net>
 * @copyright 2018 Maecia
 * @link      http://www.maecia.com/
 */
class ListNormalizer extends NormalizerBase implements UuidReferenceInterface, DenormalizerInterface
{
    /**
     * {@inheritdoc}
     */
    protected $supportedInterfaceOrClass = ListInterface::class;

    /**
     * @param array $data
     *
     * @return array|string
     */
    public function getUuid($data)
    {
        // $data is inside uuid
        return $data;
    }

    /**
     * @var UuidResolver
     */
    protected $entityResolver;

    public function __construct()
    {
        $entityRepository        = \Drupal::service('entity.repository');
        $this->entityResolver = new UuidResolver($entityRepository);
    }

    /**
     * {@inheritdoc}
     */
    public function normalize($object, $format = null, array $context = [])
    {
        $attributes  = [];
        $cardinality = $object->getFieldDefinition()->getFieldStorageDefinition()->getCardinality();
        $type        = $object->getFieldDefinition()->getFieldStorageDefinition()->getType();
        $name        = $object->getFieldDefinition()->getFieldStorageDefinition()->getName();

        if (($cardinality === 1 && $type != 'entity_reference') || $name == 'uid') {
            return $this->serializer->normalize($object[0], $format, $context);
        }

        foreach ($object as $fieldItem) {
            $attributes[] = $this->serializer->normalize($fieldItem, $format, $context);
        }

        return $attributes;
    }

    /**
     * {@inheritDoc}
     */
    public function denormalize($data, $class, $format = null, array $context = [])
    {
        if (!isset($context['target_instance'])) {
            throw new \InvalidArgumentException(
                '$context[\'target_instance\'] must be set to denormalize with the FieldItemNormalizer'
            );
        }

        if ($context['target_instance']->getParent() == null) {
            throw new \InvalidArgumentException(
                'The field item passed in via $context[\'target_instance\'] must have a parent set.'
            );
        }

        $item = $context['target_instance'];
        $type = $item->getFieldDefinition()->getType();

        if ($item instanceof EntityReferenceFieldItemList) {
            $fieldDefinition = $item->getFieldDefinition();
            $entityType      = $fieldDefinition->getSetting('target_type');
            switch ($type) {
                case 'file':
                case 'image':
                    $this->serializer->denormalize($data, FileItem::class, $format, $context);
                    break;
                case 'entity_reference':
                default:
                    try {
                        if (!is_array($data)) {
                            $data = [$data];
                        }

                        foreach ($data as $k => $id) {
                            if ($id === null) {
                                continue;
                            }
                            // $id is inside uuid we need to find uuid_host then drupal id
                            $type = $entityType;

                            if (in_array($type, ['node', 'paragraph'])) {
                                $handler = $fieldDefinition->getSetting('handler_settings');
                                $type    = array_pop($handler['target_bundles']);
                                if ($type === null && $entityType === 'paragraph') {
                                    // Note: for some reasons, question field get empty target_bundles TODO: check why ?!
                                    $type = $fieldDefinition->getTargetBundle();
                                }
                            }

                            if ($type === 'menu_link_content') {
                                $type = str_replace('field_', '', $fieldDefinition->getName());
                            }

                            if ($entityType == 'user') {
                                $type = 'users';
                            } elseif ($entityType == 'comment') {
                                $type = 'comments';
                            }

                            $domain = $entityType === 'paragraph' ? 'sections' : 'contents';

                            $query = call_user_func(
                                'Inside\\Content\\Models\\' . Str::studly($domain) . '\\' . Str::studly($type) . '::query'
                            );

                            try {
                                $content = $query->findOrFail($id);
                            } catch (ModelNotFoundException $exception) {
                                \Log::error(
                                    '[ListNormalizer::denormalize] failed to load entity reference inside id [' . $id
                                    . '] type (' . $type . ')'
                                );
                            }
                            $data[$k] = $this->entityResolver->resolve($this, $content->uuid_host, $entityType);
                        }

                        if (isset($data)) {
                            $target = [];
                            foreach ($data as $id) {
                                $target[] = ['target_id' => $id];
                            }
                            $item->setValue($target);
                        }
                    } catch (\Exception $exception) {
                        // FieldConfig
                        Log::error('[ListNormalizer::denormalize] failed on entity_reference/default => ' . $exception->getMessage());
                        //                        dd($e->getMessage());
                        //                        dump($data);
                        //                        dd($e);
                    }
            }
        } elseif ($item instanceof FieldItemList) {
            switch ($type) {
                case 'link':
                    $this->serializer->denormalize($data, LinkItem::class, $format, $context);
                    break;
                case 'language':
                    $this->serializer->denormalize($data, LanguageItem::class, $format, $context);
                    break;
                case 'created':
                case 'changed':
                case 'timestamp':
                    $this->serializer->denormalize($data, TimestampItem::class, $format, $context);
                    break;
                case 'image':
                case 'file':
                    $this->serializer->denormalize($data, FileFieldItemList::class, $format, $context);
                    break;
                case 'boolean':
                    $data = (bool)$data;
                    $this->serializer->denormalize($data, FieldItemInterface::class, $format, $context);
                    break;
                case 'uuid':
                case 'string':
                case 'integer':
                default:
                    $this->serializer->denormalize($data, FieldItemInterface::class, $format, $context);
                    break;
            }
        } else {
            // Should never happens!
            Log::error('[denormalize] wrong interface for item [' . json_encode($item) . ']');
        }
    }
}
