<?php

namespace Inside\Host\Normalizer\Field;

use Carbon\Carbon;
use Drupal;
use Drupal\Core\File\FileSystem;
use Drupal\Core\File\FileUrlGenerator;
use Drupal\file\Entity\File;
use Drupal\file\FileInterface;
use Drupal\file\Plugin\Field\FieldType\FileItem;
use Drupal\image\Plugin\Field\FieldType\ImageItem;
use Drupal\serialization\Normalizer\NormalizerBase;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use InvalidArgumentException;
use Symfony\Component\Serializer\Normalizer\DenormalizerInterface;
use Symfony\Component\Serializer\Normalizer\NormalizerInterface;

/**
 * File Normalizer
 *
 * @category  Class
 * @package   Inside\Host\Normalizer\Field
 * @author    Maecia <technique@maecia.com>
 * @copyright Maecia
 * @link      http://www.maecia.com/
 */
class FileNormalizer extends NormalizerBase implements DenormalizerInterface, NormalizerInterface
{
    protected $supportedInterfaceOrClass = [
        FileItem::class,
        ImageItem::class,
    ];

    public function normalize($object, $format = null, array $context = [])
    {
        Drupal::service('inside');

        $value = $object->getValue();
        $uri = method_exists($object, 'getFileUri') ? $object->getFileUri() : null;

        if (isset($value['target_id'])) {
            $file = $object->get('entity')->getValue();
            if (!$file) {
                return '';
            }
            $uri = $file->getFileUri();
        }

        $fileUrlGenerator = Drupal::service('file_url_generator');
        $uri = $fileUrlGenerator->generateAbsoluteString($uri);

        $path = str_replace(DIRECTORY_SEPARATOR, '/', urldecode($fileUrlGenerator->transformRelative($uri)));
        $insideStoragePath =
            str_replace('vendor/maecia/inside/', '', str_replace(DIRECTORY_SEPARATOR, '/', storage_path('app')));

        // Note: sometimes drupal is run from vendor/maecia/inside for some reason, and it uses storage/app from this
        // folder that is not correct !
        $oldInsideStoragePath = str_replace(
            DIRECTORY_SEPARATOR,
            '/',
            cms_base_path('vendor/maecia/inside/storage/app/')
        );

        return trim(
            str_replace(
                [
                    $insideStoragePath,
                    $oldInsideStoragePath,
                    storage_path('app'),
                    config('app.app_storage_path', storage_path('app')),
                    'vendor/maecia/inside/',
                    ':',
                    './',
                    'system/files/',
                    'http//',
                ],
                [
                    '',
                    '',
                    '',
                    '',
                    '',
                    '',
                    '/',
                    '',
                    '',
                ],
                $path
            ),
            '/'
        );
    }

    public function denormalize($data, $type, $format = null, array $context = [])
    {
        if (!isset($context['target_instance'])) {
            throw new InvalidArgumentException(
                '$context[\'target_instance\'] must be set to denormalize with the FieldItemNormalizer'
            );
        }

        if ($context['target_instance']->getParent() == null) {
            throw new InvalidArgumentException(
                'The field item passed in via $context[\'target_instance\'] must have a parent set.'
            );
        }

        $item = $context['target_instance'];

        if ($wrapper = Drupal::service('stream_wrapper_manager')->getViaUri('public://')) {
            $public = Drupal::service('file_url_generator')->transformRelative($wrapper->getExternalUrl());

            $uri = Str::replaceFirst($public, 'public://', $data);

            if (!str_contains($uri, 'public://')) {
                $uri = 'public://'.$uri;
            }
            $files = Drupal::entityTypeManager()->getStorage('file')->loadByProperties(['uri' => $uri]);

            if (!empty($files)) {
                foreach ($files as $i => $file) {
                    $item->setValue([$i => ['target_id' => $file->id()]]);
                }
            } else {
                $originalData = $data;
                $storage = Storage::disk('local');

                if ($storage->exists($data)) {
                    // Get user to set owner
                    $user = Drupal::currentUser();

                    $fileInfo = pathinfo($storage->path($data));

                    $storagePath = $storage->path('');
                    $folder = Carbon::today()->format('Y-m');
                    $filePath = $folder.'/'.str_replace(' ', '-', $fileInfo['filename']).'.'.Str::lower($fileInfo['extension']);
                    $filePath = Drupal::service('file_system')
                        ->getDestinationFilename($storage->path($filePath),FileSystem::EXISTS_RENAME);
                    // Use system path for Storage
                    $filePath = str_replace('/', DIRECTORY_SEPARATOR, $filePath);

                    // If file already exists, $filePath has change and does not match $storageFilePath anymore
                    $storageFilePath = str_replace($storagePath, '', $filePath);

                    if (!$storage->exists($folder)) {
                        $storage->makeDirectory($folder);
                    }

                    if ($storage->exists($storageFilePath)) {
                        $storage->delete($storageFilePath);
                    }
                    $storage->copy($data, $storageFilePath);

                    $file = File::create(
                        [
                            // Use / path for drupal files
                            'uri' => str_replace(
                                DIRECTORY_SEPARATOR,
                                '/',
                                str_replace($storagePath, 'public://', $filePath)
                            ),
                            'uid' => $user->id(),
                            'status' => FileInterface::STATUS_PERMANENT,
                        ]
                    );

                    $file->save();

                    // File is saved at this point, temporary chunk file can be deleted
                    $matches = [];
                    $storage->delete($data);
                    if (preg_match('#((chunks|fakes)/[^/]+)/[^/]+#', $originalData, $matches) === 1
                        && empty($storage->listContents($matches[1]))
                    ) {
                        $storage->deleteDir($matches[1]);
                    }

                    $item->setValue([0 => ['target_id' => $file->id()]]);
                };
            }
        }
    }
}
