<?php

namespace Inside\Host\Normalizer\Field;

use Carbon\Carbon;
use Drupal\Core\Field\Plugin\Field\FieldType\ChangedItem;
use Drupal\Core\Field\Plugin\Field\FieldType\CreatedItem;
use Drupal\Core\Field\Plugin\Field\FieldType\TimestampItem;
use Drupal\serialization\Normalizer\NormalizerBase;
use Exception;
use Illuminate\Support\Facades\Log;
use InvalidArgumentException;
use Symfony\Component\Serializer\Normalizer\DenormalizerInterface;
use Symfony\Component\Serializer\Normalizer\NormalizerInterface;

/**
 * Date Normalizer
 *
 * @category  Class
 * @package   Inside\Host\Normalizer\Field\DateNormalizer
 * @author    Maecia <technique@maecia.com>
 * @author    Nicolas Deniaud <contact@feldoe.net>
 * @copyright Maecia
 * @link      http://www.maecia.com/
 */
class DateNormalizer extends NormalizerBase implements DenormalizerInterface, NormalizerInterface
{
    protected $supportedInterfaceOrClass = [
        CreatedItem::class,
        ChangedItem::class,
        TimestampItem::class,
    ];

    public function normalize($object, $format = null, array $context = []): float|array|\ArrayObject|bool|int|string|null
    {
        $value = $object->getValue();

        if (!isset($value['value'])) {
            throw new InvalidArgumentException(
                '$value[\'value\'] must be set to denormalize with the DateFieldNormalizer'
            );
        }

        try {
            $date = date('Y-m-d H:i:s', (int)$value['value'] ?? time());
        } catch (Exception $e) {
            Log::error(
                '[DateFieldNormalizer] failed to normalize date {'.$value['value'].'} ['.$e->getMessage().']'
            );
            $date = null;
        }

        return $date;
    }

    public function denormalize($data, $class, $format = null, array $context = [])
    {
        if (!isset($context['target_instance'])) {
            throw new InvalidArgumentException(
                '$context[\'target_instance\'] must be set to denormalize with the FieldItemNormalizer'
            );
        }

        if ($context['target_instance']->getParent() == null) {
            throw new InvalidArgumentException(
                'The field item passed in via $context[\'target_instance\'] must have a parent set.'
            );
        }

        $item = $context['target_instance'];
        if ($this->isValidTimeStamp($data)) {
            $item->setValue((int)$data);
            return;
        }
        try {
            // Data is a string representing a date/time in format Y-m-d H:i:s
            $date = Carbon::createFromFormat('Y-m-d H:i:s', $data);
        } catch (Exception $e) {
            Log::error('[DateFieldNormalizer] failed to denormalize ['.$e->getMessage().'] {'.$data.'}');
            $date = Carbon::now();
        }

        $item->setValue($date->getTimestamp());
    }
    protected function isValidTimeStamp(string $timestamp): bool
    {
        return ((string) (int) $timestamp === $timestamp)
            && ($timestamp <= PHP_INT_MAX)
            && ($timestamp >= ~PHP_INT_MAX);
    }
}
