<?php

namespace Inside\Host\Normalizer\Entity;

use Exception;
use Illuminate\Support\Facades\Log;

/**
 * Node Normalizer
 *
 * @category  Class
 * @package   Inside\Host\Normalizer\Entity\NodeNormalizer
 * @author    Maecia <technique@maecia.com>
 * @author    Nicolas Deniaud <contact@feldoe.net>
 * @copyright 2018 Maecia
 * @link      http://www.maecia.com/
 */
class NodeNormalizer extends BaseEntityNormalizer
{

    /**
     * {@inheritdoc}
     */
    protected $removableFields = [
        'nid',
        'vid',
        'type',
        'revision_timestamp',
        'revision_uid',
        'revision_log',
        'promote',
        'sticky',
        'default_langcode',
        'revision_default',
        'revision_translation_affected',
        'path',
        'revision_uuidid',
        'weight',
        'width',
        'height',
        'alt',
        'roles',
        'preferred_admin_langcode',
        'preferred_langcode',
        'pass',
        'timezone',
        'access',
        'init',
        'content_translation_outdated',
        'content_translation_source',
    ];

    /**
     * {@inheritDoc}
     */
    protected $swappableNames = [
        'created' => 'created_at',
        'changed' => 'updated_at',
        'uid' => 'author',
        'uuid' => 'uuid_host'
    ];


    /**
     * {@inheritDoc}
     */
    public function normalize($entity, $format = null, array $context = [])
    {
        if (!isset($context['langcode']))  {
            $context['langcode'] = $entity->language()->getId();
        }
        $attributes = parent::normalize($entity, $format, $context);

        $attributes = $this->formatDatasToNormalization($entity, $attributes);

        // Add Type and Bundle to be able to desarlize it later
        $attributes['type'] = $entity->getEntityTypeId();
        $attributes['bundle'] = $entity->bundle();
        $attributes['uuid'] = get_lumen_entity_uuid($entity);

        return $attributes;
    }


    /**
     * {@inheritDoc}
     * @throws Exception
     */
    public function denormalize($data, $class, $format = null, array $context = [])
    {
        if (!isset($data['type'])) {
            throw new Exception('The type must be specified.');
        }

        $values = [];
        $type = $data['type']; // Only used to locate correct entity normalizer ( means been here )

        $entity_type = $this->getEntityTypeDefinition($type);
        $default_langcode_key = $entity_type->getKey('default_langcode');
        $langcode_key = $entity_type->getKey('langcode');


        $data = $this->prepareLangs($data, $values, $default_langcode_key, $langcode_key);

        // We need to get the correct bundle key for this node type
        // There is usually a swap between bundle and type
        // This is an important part, Bundle won't be recognized by entity storage
        if ($entity_type->hasKey('bundle')) {
            $bundle_key = $entity_type->getKey('bundle');
            $values[$bundle_key] = $data['bundle'];
            // Unset the bundle key from data, if it's there.
            unset($data[$bundle_key]);
        }

        $bundle = $data['bundle'];

        unset($data['bundle']);
        unset($data['entity_type']);

        try {
            $entity = $this->createEntity($values, $data, $context, 'node', 'nid');
            $result = $this->formatDatasToDenormalization($data, 'node', $bundle);

            $this->denormalizeFieldData($result, $entity, $format, $context);
        } catch (Exception $e) {
            Log::error('[NodeNormalizer] Node denormalization failed [' . $e->getMessage() . ']');
            Log::debug('[NodeNormalizer] Node denormalization failed trace [' . $e->getTraceAsString() . ']');
        }

        return $entity;
    }
}
