<?php

namespace Inside\Host\Normalizer\Entity;

use Drupal;
use Drupal\Core\Entity\EntityInterface;
use Drupal\serialization\Normalizer\EntityNormalizer;
use Log;

/**
 * Class ContentEntityNormalizer
 *
 * This is our base Content Entity Normalizer
 * This should not be used directly
 *
 * @package Inside\Host\Normalizer
 */
abstract class BaseEntityNormalizer extends EntityNormalizer
{

    use FieldableEntityNormalizer;

    /**
     * Unwanted fields
     *
     * @var array
     */
    protected $removableFields = [];

    /**
     * Conversion between Drupal Names => Inside 2 Names
     *
     * @var array
     */
    protected $swappableNames = [];

    /**
     * Format normalized data to inside 2 style
     *
     * @param mixed $entity
     * @param array $attributes
     *
     * @return array
     */
    protected function formatDatasToNormalization($entity, array $attributes): array
    {
        $result = [];

        foreach ($attributes as $name => $value) {
            if (!in_array($name, $this->removableFields)) {
                $name = str_replace('field_', '', $name);
                $name = $this->swappableNames[$name] ?? $name;

                if ($name == 'pid') {
                    if (is_array($value)) {
                        $value = empty($value) ? null : $value[0];
                    }
                    if (is_string($value)) {
                        $exploded = explode(':', $value);
                        $pid      = $exploded[1] ?? null;

                        if ($pid) {
                            $parent = Drupal::service('entity.repository')->loadEntityByUuid(
                                $entity->getEntityTypeId(),
                                $pid
                            );
                            if ($parent->hasTranslation($entity->language()->getId())) {
                                $parent = $parent->getTranslation($entity->language()->getId());
                            }
                            $value = get_lumen_entity_uuid($parent);
                        }
                    }
                }

                $result[$name] = $value;
            }
        }

        return $result;
    }

    /**
     * Get denormalizable data from inside 2 datas
     *
     * @param array  $attributes Inside 2 style datas
     * @param string $type       Entity Type
     * @param string $bundle     Entity Bundle
     *
     * @return array
     */
    protected function formatDatasToDenormalization(array $attributes, $type, $bundle): array
    {
        $result = [];

        foreach ($attributes as $name => $value) {
            $name = array_flip($this->swappableNames)[$name] ?? $name;

            if ($this->isCustom($type, $bundle, $name)) {
                $name = 'field_' . $name;
            }

            $result[$name] = $value;
        }

        return $result;
    }

    /**
     * Does this field is a custom one ?
     *
     * @param string $type      Entity Type
     * @param string $bundle    Entity Bundle
     * @param string $fieldName Field name
     *
     * @return bool
     */
    private function isCustom(string $type, string $bundle, string $fieldName): bool
    {
        return in_array(
            'field_' . $fieldName,
            array_keys(Drupal::service('entity_field.manager')->getFieldDefinitions($type, $bundle))
        );
    }

    /**
     * Create a fresh entity with minimum $values sets of $type and prepare it for denormalization
     *
     * @param array  $values  Minimum values to create a fresh entity ( eg: bundle => 'news' )
     * @param array  $data    Data to be denormalized
     * @param array  $context Denormalization context
     * @param string $type    Entity Type
     * @param string $idName  Id field name ( eg: nid for a node, cid for a comment, etc... )
     *
     * @return EntityInterface Fresh entity
     * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
     * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
     * @throws \Drupal\Core\Entity\EntityStorageException
     */
    protected function createEntity(
        array $values, array &$data, array &$context, string $type, string $idName
    ): EntityInterface {
        $entity = Drupal::entityTypeManager()->getStorage($type)->create($values);

        $context['entity'] = $entity;

        if (array_key_exists('uuid_host', $data)) {
            // On update, we wan't to get uid to properly save entity in drupal
            $entity = Drupal::service('entity.repository')->loadEntityByUuid($type, $data['uuid_host']);

            if ($entity && $entity->isTranslatable() && isset($data['langcode'])) {
                if (!$entity->hasTranslation($data['langcode'])) {
                    $entity = $entity->addTranslation($data['langcode'], $entity->toArray());
                } else {
                    $entity = $entity->getTranslation($data['langcode']);
                }
                // We have the entity in the correct language, we don't need the langcode anymore
                unset($data['langcode']);
            }

            if ($entity) {
                // We got an object with this uuid and type, so it's un update !
                $entity->enforceIsNew(false);
                $data[$idName] = $entity->id(); // Update needs id
            } else {
                Log::error(
                    "[BaseEntityNormalizer::createEntity] FAILED TO LOAD existing object $type (" . $data['uuid_host']
                    . ")"
                );

                // Item is in inside but not in drupal anymore... let's create it again
                $entity = Drupal::service('entity_type.manager')->getStorage($type)->create($values);

                // Relink
                $data['uuid_host'] = $entity->uuid();
            }
        }

        return $entity;
    }

    /**
     * Prepare languages stuffs
     *
     * @param array  $data               Data to be denormalized
     * @param array  $values             Minimum values to create a fresh entity
     * @param string $defaultLangcodeKey Default langcode key
     * @param string $langcodeKey        Wanted langcode key
     *
     * @return array Modified data to be denormalized with correct setted language config
     */
    protected function prepareLangs(array $data, array &$values, string $defaultLangcodeKey, string $langcodeKey): array
    {
        if (isset($data[$defaultLangcodeKey])) {
            foreach ($data[$defaultLangcodeKey] as $item) {
                if (!empty($item['value']) && isset($item['lang'])) {
                    $values[$langcodeKey] = $item['lang'];
                    break;
                }
            }
            unset($data[$defaultLangcodeKey]);
        }
        if (count(list_languages()) > 1) {
            if (isset($data['uuid_host'])) {
                $entity = Drupal::service('entity.repository')->loadEntityByUuid('node', $data['uuid_host']);
                if ($entity && (count($entity->getTranslationLanguages()) >= 1)) {
                    $data['content_translation_source'] = $entity->language()->getId();
                }
            } else {
                // We got a new content, let's assume that this content is the source translation!
                $data['content_translation_source'] = $data['langcode'] ?? config('app.locale');
            }
        }

        return $data;
    }
}
