<?php

namespace Inside\Host\Migrations;

use Illuminate\Support\Arr;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use Inside\Content\Facades\Schema;
use Inside\Host\Bridge\BridgeContentType;
use Inside\Layout\Models\Layout;
use Inside\Layout\Models\LayoutPath;
use Symfony\Component\Console\Formatter\OutputFormatterStyle;
use Symfony\Component\Console\Input\ArrayInput;
use Symfony\Component\Console\Output\ConsoleOutput;

trait CreateContentTypes
{
    protected $console = null;

    protected $name = null;

    /** @var BridgeContentType $bridge */
    protected $bridge = null;

    protected $booted = false;

    protected $revision = null;

    protected $file = null;

    public function bootUp($file, $name = null)
    {
        $this->file = $file;

        // Get name
        if ($name) {
            $this->name = $name;
        } else {
            $this->name = str_replace('_', ' ', Str::snake(get_class($this)));
        }

        $this->console = $this->getConsole();

        $this->bridge = new BridgeContentType($this->console);

        $matches = [];
        if (preg_match('/^(\d{4}_\d{2}_\d{2}_\d{6})_.*\.php$/', basename($file), $matches)) {
            $this->revision = $matches[1];
        }

        $this->booted = true;
    }

    public function getConsole(): ConsoleOutput
    {
        if (!$this->console) {
            // Prepare a console
            $this->console = new ConsoleOutput;
            $outputStyle   = new OutputFormatterStyle('white', 'green', ['bold', 'underscore']);
            $this->console->getFormatter()->setStyle('module', $outputStyle);
            $outputStyle = new OutputFormatterStyle('yellow', 'blue', ['bold', 'underscore']);
            $this->console->getFormatter()->setStyle('class', $outputStyle);
            $outputStyle = new OutputFormatterStyle('black', 'green');
            $this->console->getFormatter()->setStyle('success', $outputStyle);
            $outputStyle = new OutputFormatterStyle('black', 'yellow');
            $this->console->getFormatter()->setStyle('warning', $outputStyle);
            $outputStyle = new OutputFormatterStyle('white', 'red');
            $this->console->getFormatter()->setStyle('error', $outputStyle);
            $outputStyle = new OutputFormatterStyle('yellow');
            $this->console->getFormatter()->setStyle('note', $outputStyle);

            if (!empty($this->name)) {
                $this->console->writeln(
                    "<info>Début de l'installation des types de contenu du package</info> <module>$this->name</module>"
                );
            }
        }

        return $this->console;
    }

    public function writeResult(bool $success): void
    {
        if ($success) {
            $this->getConsole()->writeln(' <fg=green>✔</fg=green>');

            return;
        }
        $this->getConsole()->writeln(' <fg=red>✘</fg=red>');
    }

    public function separatorLine(): void
    {
        $this->writeln(str_repeat('-', 80));
    }


    public function getBridge(): ?BridgeContentType
    {
        return $this->bridge;
    }

    public function migrateContent($name, bool $deleteBefore = false, $domain = 'contents')
    {
        $console = $this->getConsole();
        if (!$this->booted) {
            $console->writeln("<error>La migration n'est pas démarrée</error>");
            exit(-1);
        }
        $contentTypeFile = dirname($this->file) . '/../' . $domain . '/' . $name . '/' . $this->revision . '.php';

        if (!File::exists($contentTypeFile)) {
            $console->writeln(
                "<error>Le fichier descriptif du type de " . ($domain === 'contents' ? 'contenu' : 'section') . " n'a pu être trouvé [$contentTypeFile]</error>"
            );
            exit(-1);

        }
        try {
            $contentDescription = include_once $contentTypeFile;

            if ($deleteBefore && Schema::hasModel($contentDescription['options']['name'])) {

                $console->writeln(
                    "<info>Suppression du type de " . ($domain === 'contents' ? 'contenu' : 'section') . "</info> <class>" . Str::studly(
                        $contentDescription['options']['name']
                    ) . "</class>"
                );
                $this->bridge->contentTypeDelete($contentDescription['options']['name'], $domain);
            }

            $console->writeln(
                "<info>Création du type de " . ($domain === 'contents' ? 'contenu' : 'section') . "</info> <class>" . Str::studly($contentDescription['options']['name'])
                . "</class>"
            );

            $this->bridge->contentTypeUpdateOrCreate(
                $contentDescription['options']['name'],
                $contentDescription['options'],
                $contentDescription['fields'],
                $contentDescription['groups'] ?? [],
                $domain
            );
            $console->writeln(
                "<comment>Type de contenu <class>" . Str::studly($contentDescription['options']['name'])
                . "</class> correctement créé</comment> "
            );
        } catch (\Exception $e) {
            $console->writeln(
                "<error>La class <class>" . Str::studly($contentDescription['options']['name'])
                . "</class> n'a pu être créée/mise à jour *" . $e->getMessage() . "*</error>"
            );
            exit(-1);
        }
        Schema::refresh();
    }

    public function finishUp()
    {
        $console = $this->getConsole();
        if (!$this->booted) {
            $console->writeln("<error>La migration n'est pas démarrée</error>");
            exit(-1);
        }
        if (isset($this->layouts) && is_array($this->layouts) && !empty($this->layouts)) {
            $console->writeln("<info>Installation des pages</info>");
            foreach ($this->layouts as $layout => $config) {
                $this->installLayout($layout, $config['paths'], $config['config']);
            }
        } else {
            $console->writeln("Aucune page à installer");
        }

        $console->writeln("<success>Installation terminées</success>");
    }

    protected function installLayout($name, array $paths, array $layoutConfig)
    {
        $console = $this->getConsole();
        if (!$this->booted) {
            $console->writeln("<error>La migration n'est pas démarrée</error>");
            exit(-1);
        }
        $console->writeln("<info>Installation de la page <class>$name</class></info>");
        if (($path = LayoutPath::where('path', Arr::first($paths))->first()) != null) {
            $layout = $path->layout;
            $layout->update($layoutConfig);
        } else {
            $layout = Layout::create($layoutConfig);
            foreach ($paths as $langcode => $path) {
                if (in_array($langcode, list_languages())) {
                    $layoutPath = new LayoutPath;
                    $layoutPath->fill(
                        [
                            'path'     => $path,
                            'langcode' => $langcode,
                        ]
                    )->layout()->associate($layout)->save();
                }
            }
        }
        $console->writeln("<comment>Page <class>$name</class> correctement installés</comment>");
    }

    public function call($command, array $arguments = [])
    {
        $arguments['command'] = $command;

        return (new \Illuminate\Console\Application(app(), app('events'), app()->version()))->find($command)->run(
            $this->createInputFromArguments($arguments), $this->getConsole()
        );
    }

    protected function createInputFromArguments(array $arguments)
    {
        return tap(new ArrayInput($arguments), function ($input) {
            if ($input->hasParameterOption(['--no-interaction'], true)) {
                $input->setInteractive(false);
            }
        });
    }

    public function bootDown($file, $name = null)
    {
        // TODO
    }

    public function finishDown()
    {
        // TODO
    }
}
