<?php

namespace Inside\Host\Helpers\Adapter\Form;

use Illuminate\Support\Str;
use Inside\Content\Models\Contents\Users;
use Inside\Content\Models\Model;

/**
 * Flatten formDisplay from Drupal 8
 *
 * @category  Class
 * @package   Inside\Host\Adapter\Form\UserForm
 * @author    Maecia <technique@maecia.com>
 * @copyright 2018 Maecia
 * @link      http://www.maecia.com/
 */
class UserForm extends BaseForm
{
    /**
     * {@inheritDoc}
     */
    protected const REMOVABLE_FIELD = [
        'timezone',
        'translation',
        'account',
        'uid',
        'uuid',
        'preferred_langcode',
        'preferred_admin_langcode',
        'access',
        'login',
        'init',
        'roles',
        'default_langcode',
        'content_translation_source',
        'content_translation_outdated',
        'content_translation_uid',
        'content_translation_status',
        'content_translation_created',
    ];

    /**
     * {@inheritDoc}
     */
    protected const FIELD_NAME = [
        'language' => 'langcode',
    ];

    /**
     * {@inheritDoc}
     */
    protected const TABLE_NAME = [
        'user' => 'users',
    ];

    /**
     * {@inheritDoc}
     */
    protected function getModel(): string
    {
        $bundle = $this::TABLE_NAME[$this->bundle] ?? $this->bundle;

        return 'Inside\Content\Models\Contents\\' . Str::studly($bundle);
    }

    /**
     * {@inheritDoc}
     */
    protected function handleCustomFields(&$oldFields): void
    {
        $model = $this->getModel();

        $customs = [
            ['name' => 'name', 'type' => 'text', 'title' => 'Username', 'widget' => 'text', 'default' => null],
            ['name' => 'email', 'type' => 'email', 'title' => 'Email', 'widget' => 'email', 'default' => null],
            [
                'name'    => 'password',
                'type'    => 'password',
                'title'   => 'Password',
                'widget'  => 'password',
                'default' => null,
            ],
            ['name' => 'status', 'type' => 'checkbox', 'title' => 'Status', 'widget' => 'onoff', 'default' => 1],
        ];

        foreach ($customs as $custom) {
            $title       = [];
            $description = [];

            foreach ($this->languages as $language) {
                $title[$language]       = (string)t($custom['title'], [], ['langcode' => $language]);
                $description[$language] = '';
            }

            $options = [
                'title'               => $title,
                'description'         => $description,
                'cardinality'         => 1,
                'translatable'        => false,
                'required'            => true,
                'default'             => $custom['default'],
                'weight'              => isset($this->drupalFormFields['account']) ? $this->drupalFormFields['account']['weight'] : 0,
                'widget'              => $custom['widget'],
                'group'               => [],
                'searchable'          => true,
                'search_result_field' => true,
                'searchable_filter'   => false,
            ];

            if (in_array($custom['name'], ['password', 'status'])) {
                $options['searchable'] = false;
            } elseif (in_array($custom['name'], ['name', 'email'])) {
                // Model as already been updated get info there
                $modelInformation      = Model::where('class', Users::class)->first();
                $options['searchable'] = isset($modelInformation->options['search_users_' . $custom['name']]) ? $modelInformation->options['search_users_' . $custom['name']] == 1 : true;
            }
            if (in_array($custom['name'], ['status'])) {
                $options['searchable_filter'] = true;
            }

            $this->attachGroup('account', $options);

            $data = [
                'model'     => $model,
                'name'      => $custom['name'],
                'type'      => $custom['type'],
                'options'   => $options,
                'displayed' => true,
            ];

            $this->createOrUpdateField($data);
            if (($key = $oldFields->search($custom['name'])) !== false) {
                $oldFields->forget($key);
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    protected function handleAttributeFields(string $name, $field = null, bool $isHidden = false): void
    {
        $model     = $this->getModel();
        $saveField = false;
        $type      = 'text';
        $weight    = 0;

        if (is_array($field)) {
            $weight = $field['weight'];
        }

        $options = [
            'cardinality'         => 1,
            'translatable'        => false,
            'required'            => true,
            'weight'              => $weight,
            'default'             => null,
            'widget'              => '',
            'group'               => [],
            'searchable'          => false,
            'search_result_field' => false,
        ];

        switch ($name) {
            case 'langcode':
                $avalaibleLanguages = \Drupal::languageManager()->getLanguages();

                $saveField                 = true;
                $titleText                 = 'Language';
                $type                      = 'select';
                $options['widget']         = 'select';
                $options['default']        = \Drupal::languageManager()->getDefaultLanguage()->getId();
                $options['allowed_values'] = [];

                foreach ($avalaibleLanguages as $avalaibleLanguage) {
                    $options['allowed_values'][$avalaibleLanguage->getId()] = $avalaibleLanguage->getName();
                }
                break;
        }

        if ($saveField) {
            $title       = [];
            $description = [];

            foreach ($this->languages as $language) {
                $title[$language]       = (string)t($titleText, [], ['langcode' => $language]);
                $description[$language] = '';
            }

            $options['title']       = $title;
            $options['description'] = $description;

            $this->attachGroup($name, $options);

            $data = [
                'model'     => $model,
                'name'      => $name,
                'type'      => $type,
                'options'   => $options,
                'displayed' => !$isHidden,
            ];

            $this->createOrUpdateField($data);
        }
    }
}
