<?php

namespace Inside\Host\Exodus\Traits;

use Illuminate\Support\Collection;
use Inside\Host\Exodus\Enum\ListingTypeEnum;
use Inside\Host\Exodus\Services\ContentTypeStatusService;

trait HasThirdPartySettingsConfiguration
{
    /**
     * Check if the content type is categorizable in third party settings of inside module from Drupal.
     *
     * @return bool
     */
    public static function isCategorizable(): bool
    {
        return app(ContentTypeStatusService::class)->categorizables()->contains('type', class_to_type(static::class));
    }

    /**
     * Check if the content type is permissible in third party settings of inside module from Drupal.
     *
     * @return bool
     */
    public static function isPermissible(): bool
    {
        return app(ContentTypeStatusService::class)->permissibles()->contains('type', class_to_type(static::class));
    }

    /**
     * Check if the content type is aliasable in third party settings of inside module from Drupal.
     *
     * @return bool
     */
    public static function isAliasable(): bool
    {
        return app(ContentTypeStatusService::class)->aliasables()->contains('type', class_to_type(static::class));
    }

    /**
     * Check if the content type is global searchable in third party settings of inside module from Drupal.
     *
     * @return bool
     */
    public static function isGlobalSearchable(): bool
    {
        return app(ContentTypeStatusService::class)->globalSearchables()->contains('type', class_to_type(static::class));
    }

    /**
     * Check if the content type is searchable in third party settings of inside module from Drupal.
     *
     * @return bool
     */
    public static function isSearchable(): bool
    {
        return app(ContentTypeStatusService::class)->searchables()->contains('type', class_to_type(static::class));
    }

    /**
     * Check if the content type is translatable in third party settings of inside module from Drupal.
     *
     * @return bool
     */
    public static function isTranslatable(): bool
    {
        return app(ContentTypeStatusService::class)->translatables()->contains('type', class_to_type(static::class));
    }

    /**
     * Check if the content type listing is "content" in third party settings of inside module from Drupal.
     *
     * @return bool
     */
    public static function isContentListing(): bool
    {
        return app(ContentTypeStatusService::class)->ofListingType(ListingTypeEnum::CONTENT)->contains('type', class_to_type(static::class));
    }


    /**
     * Check if the content type listing is "taxo" in third party settings of inside module from Drupal.
     *
     * @return bool
     */
    public static function isTaxonomyListing(): bool
    {
        return app(ContentTypeStatusService::class)->ofListingType(ListingTypeEnum::TAXONOMY)->contains('type', class_to_type(static::class));
    }

    /**
     * Check if the content type listing is "hidden" in third party settings of inside module from Drupal.
     *
     * @return bool
     */
    public static function isHiddenListing(): bool
    {
        return app(ContentTypeStatusService::class)->ofListingType(ListingTypeEnum::HIDDEN)->contains('type', class_to_type(static::class));
    }

    /**
     * Get the labels of the content type from third party settings of inside module from Drupal.
     *
     * @return Collection
     */
    public static function labels(): Collection
    {
        $nodeType = app(ContentTypeStatusService::class)->firstWhere('type', class_to_type(static::class));

        return collect(data_get($nodeType, 'translations', []));
    }
}
