<?php

namespace Inside\Host\EventSubscriber\EntityType;

use Drupal\Core\Cache\Cache;
use Drupal\node\Entity\NodeType;
use Drupal\node\NodeInterface;
use Exception;
use Inside\Content\Exceptions\DynamicModelNotWritable;
use Inside\Content\Facades\DynamicClass;
use Inside\Content\Models\Model;
use Inside\Host\Event\BaseEvent;
use Inside\Host\Exceptions\ReservedTableNameException;
use Inside\Permission\Models\Permission;
use Inside\Permission\Models\PermissionSchema;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Support\Str;
use Log;
use Schema;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Base event subscriber
 *
 * @category  Class
 * @package   Inside\Host\EventSubscriber\EntityType\BaseEntityTypeEventSubscriber
 * @author    Maecia <technique@maecia.com>
 * @copyright 2018 Maecia
 * @link      http://www.maecia.com/
 */
abstract class BaseEntityTypeEventSubscriber implements EventSubscriberInterface
{

    /**
     * Prefix for each content table name
     */
    const TABLE_PREFIX = 'inside_content_';

    /**
     * Suffix for each content table name
     */
    const TABLE_SUFFIX = '';

    /**
     * Prefix for the lumen namespace
     */
    const NAMESPACE_PREFIX = 'Inside\Content\Models\Contents\\';

    /**
     * SUFFIX CLASS
     */
    const CLASS_SUFFIX = '';

    /**
     * Reserved names by the core
     */
    const RESERVED_NAME = [];

    /**
     * The table name
     *
     * @var string
     */
    protected $table = null;

    /**
     * The bundle name
     *
     * @var string
     */
    protected $bundle = null;

    /**
     * The entity options
     *
     * @var string
     */
    protected $options = null;

    /**
     * @var array|null
     */
    protected $languages = null;

    /**
     * class
     *
     * @var string|null
     */
    protected $class;

    /**
     * The entity bundle key
     *
     * @var string
     */
    protected $bundleKey = 'id';

    /**
     * Create new base entity event subscriber instance
     */
    public function __construct()
    {
        \Drupal::service('inside');

        $this->languages = array_keys(\Drupal::languageManager()->getLanguages());
    }

    /**
     * Initialize entity type values
     *
     * @param  BaseEvent  $event
     *
     * @throws ReservedTableNameException
     */
    protected function init(BaseEvent $event)
    {
        $entity = $event->getItem();
        $this->bundle = $entity->get($this->bundleKey);
        $this->table = static::TABLE_PREFIX.$this->bundle.static::TABLE_SUFFIX;
        $this->class = static::NAMESPACE_PREFIX.Str::studly($this->bundle).static::CLASS_SUFFIX;
        $this->options = $entity->getThirdPartySettings('inside');

        $this->options['translatable'] = false;

        foreach ($this->languages as $language) {
            $this->options['title'][$language] = match (get_class($entity)) {
                'Drupal\comment\Entity\CommentType', 'Drupal\paragraphs\Entity\ParagraphsType', 'Drupal\system\Entity\Menu' => t($entity->get('label'),
                    [], ['langcode' => $language]),
                default => t($entity->get('name'), [], ['langcode' => $language]),
            };
        }

        $childrenType = get_drupal_children_type($entity->getEntityTypeId());

        if ($childrenType) {
            $this->options['translatable'] =
                \Drupal::config('language.content_settings.'.$childrenType.'.'.$this->bundle)->get(
                    'language_alterable'
                );
        }

        if ($this->options['translatable']) {
            // Reload titles with translations

            // Save current language translator
            $currentLanguage = \Drupal::languageManager()->getCurrentLanguage();
            $defaultLanguage = $entity->language()->getId();
            foreach ($this->languages as $language) {
                // Here we get our translator
                $translator = \Drupal::languageManager()->getLanguage($language);
                // Override current language with the one we want
                \Drupal::languageManager()->setConfigOverrideLanguage($translator);

                if ($language != $defaultLanguage) {
                    $nodeType = NodeType::load($entity->id());
                    if ($nodeType) {
                        $this->options['title'][$language] = (string) $nodeType->get('name');
                    }
                }
            }
            // Set back to current language translator
            \Drupal::languageManager()->setConfigOverrideLanguage($currentLanguage);
        }

        $this->options['name'] = $this->bundle.static::TABLE_SUFFIX;

        if (in_array($this->table, static::RESERVED_NAME)) {
            Log::error('The table name is reserved.', ['name' => $this->table]);
            throw new ReservedTableNameException('The table name is reserved.', 400);
        }
    }

    abstract protected function createTable(): void;

    /**
     * Triggered on node type insert
     *
     * @param  BaseEvent  $event
     *
     * @throws ReservedTableNameException|DynamicModelNotWritable
     */
    public function insert(BaseEvent $event): void
    {
        $this->init($event);

        if (Schema::hasTable($this->table)) {
            return;
        }

        $this->createTable();

        Model::create(
            [
                'class' => $this->class,
                'options' => $this->options,
            ]
        );

        DynamicClass::rebuild();

        Log::info('Table '.$this->table.' successfully created', ['table' => $this->table]);
        InsideSchema::refresh();
        $this->invalidateCache($event->getItem());
    }

    /**
     * Triggered on paragraph type update
     *
     * @param  BaseEvent  $event
     *
     * @return void
     * @throws Exception
     */
    public function update(BaseEvent $event)
    {
        $this->init($event);

        $model = Model::where('class', $this->class)->first();

        if ($model === null) {
            Model::create(
                [
                    'class' => $this->class,
                    'options' => $this->options,
                ]
            );

            return;
        }

        $model->update(['options' => $this->options]);
        if (isset($this->options['searchable']) && ($this->options['searchable'] == 1)) {
            // Rebuild models in case this content type was not searchable before otherwise indexation will fail
            DynamicClass::rebuild();

            //Artisan::call('index:rebuild', ['--type' => class_to_type($this->class), '--silent' => true]);
        }
        InsideSchema::refresh();
        $this->invalidateCache($event->getItem());
    }

    /**
     * Triggered on node type delete
     *
     * @param  BaseEvent  $event
     *
     * @return void
     * @throws DynamicModelNotWritable
     * @throws ReservedTableNameException
     */
    public function delete(BaseEvent $event)
    {
        $this->init($event);

        if (!Schema::hasTable($this->table)) {
            return;
        }

        // We can't drop the users table
        if ($this->table === 'inside_content_users') {
            return;
        }

        Schema::drop($this->table);
        Model::where('class', $this->class)->delete();

        // Delete permission as well
        Permission::where('type', $this->class)->delete();
        PermissionSchema::where('authorizable_type', $this->class)->delete();

        DynamicClass::rebuild();
        InsideSchema::refresh();
        $this->invalidateCache($event->getItem());

        Log::info('Table '.$this->table.' successfully deleted', ['table' => $this->table]);
    }

    /**
     * @param NodeType $entity
     */
    protected function invalidateCache($entity)
    {
        Cache::invalidateTags($entity->getCacheTagsToInvalidate());
    }
}
