<?php

namespace Inside\Host\Commands;

use Drupal\comment\Entity\Comment;
use Drupal\comment\Entity\CommentType;
use Drupal\Component\Serialization\Yaml;
use Drupal\Component\Utility\Random;
use Drupal\Component\Uuid\Uuid;
use Drupal\Core\Entity\Entity;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\file\Entity\File;
use Drupal\menu_item_extras\Entity\MenuItemExtrasMenuLinkContent;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\paragraphs\Entity\ParagraphsType;
use Drupal\system\Entity\Menu;
use Drupal\user\Entity\User;
use Drush\Commands\DrushCommands;
use Illuminate\Support\Str;
use Inside;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;
use Inside\Host\Bridge\BridgeContentType;

/**
 * Inside specific drush commands
 *
 * @category Class
 * @package  Inside\Host\Commands\InsideDrushCommands
 * @author   Maxime Jacquot <maxime.jacquot@maecia.com>
 * @copyright 2018 Maecia
 * @link     http://www.maecia.com/
 */
class InsideDrushCommands extends DrushCommands
{
    /**
     * Test functionality
     *
     * @command inside:test
     * @return void
     *
     * @usage inside:test
     */
    public function test(): void
    {
        \Drupal::service('inside');

        try {
            // Make your test here !
            $name = 'Test';
            $fields = [

            ];
            $type = new NodeType([
                'type' => Str::snake($name),
                'name' => $name,
                'new_revision' => 'revision',
            ], 'node_type');

            $type->enforceIsNew();
            $status = $type->save();
        } catch (\Exception $e) {
            var_dump($e->getMessage(), $e->getTrace());
        }
    }

    /**
     * Resave all content types and fields configuration.
     *
     * @command inside:resave-form-displays
     * @aliases rfd
     * @usage inside:save-content-types
     * @return void
     * @throws EntityStorageException
     */
    public function resave_form_displays(): void
    {
        $entityTypeManager = \Drupal::service('entity_type.manager');

        $types = [
            'node' => [],
            'user' => ['user'],
            'comment' => [],
            'paragraph' => [],
            'menu_link_content' => [],
        ];

        // Node
        $nodeTypes = NodeType::loadMultiple();
        foreach ($nodeTypes as $nodeType) {
            $this->output()->writeln('Saving node type ' . $nodeType->id());

            $nodeType->save();

            $types['node'][] = $nodeType->id();
        }

        // Paragraph
        $paragraphTypes = ParagraphsType::loadMultiple();
        foreach ($paragraphTypes as $paragraphType) {
            $this->output()->writeln('Saving paragraph type ' . $paragraphType->id());

            $paragraphType->save();

            $types['paragraph'][] = $paragraphType->id();
        }

        // Menus
        $menus = Menu::loadMultiple();
        foreach ($menus as $menu) {
            if (in_array($menu->id(), ['account', 'tools', 'admin'])) {
                continue;
            }

            $this->output()->writeln('Saving menu type ' . $menu->id());

            $menu->save();

            $types['menu_link_content'][] = $menu->id();
        }

        // Comments
        $commentTypes = CommentType::loadMultiple();
        foreach ($commentTypes as $commentType) {
            $this->output()->writeln('Saving comment type ' . $commentType->id());

            $commentType->save();

            $types['comment'][] = $commentType->id();
        }

        // Users
        $this->output()->writeln('Saving user type user');
        $model = \Inside\Content\Models\Model::where('class', 'Inside\Content\Models\Contents\Users')->first();

        if ($model) {
            $options = $model->options;
            $options['translatable'] = \Drupal::config('language.content_settings.user.user')->get('language_alterable');
            $model->update(['options' => $options]);
        }

        foreach ($types as $type => $bundles) {
            foreach ($bundles as $bundle) {
                $id = $type . '.' . $bundle . '.default';

                $this->output()->writeln('Saving form display ' . $id);

                $formDisplay = $entityTypeManager->getStorage('entity_form_display')->load($id);

                if (!$formDisplay) {
                    $this->output()->writeln('Form display ' . $id . 'was not found, skipping.');
                    continue;
                }

                $formDisplay->save();
            }
        }
    }

    /**
     * Delete all content types
     *
     * @command inside:clean-type-content
     * @usage inside:clean-type-content
     * @return void
     */
    public function clean_type_content($options = []): void
    {
        new \Inside\Host\Services\InsideService;
        $bridge = new BridgeContentType;
        $models = \Inside\Content\Models\Model::all();

        foreach ($models as $model) {
            if ($model->class === 'Inside\Content\Models\Contents\Users') {
                continue;
            }
            $bridge->contentTypeDelete(class_to_type($model->class));
        }

        $this->output()->writeln("all content types have been removed");
        $this->output()->writeln('Removing deleted fields tables');
        field_purge_batch(500);
    }

    /**
     * Purge fields
     *
     * @command inside:field-purge
     * @usage inside:field-purge
     * @return void
     */
    public function clean_drupal_fields()
    {
        $limit = \Drupal::config('field.settings')->get('purge_batch_size');
        field_purge_batch($limit);
        $this->io()->text("<info>✔</info> Tables Drupal nettoyées</info>");
    }

    /**
     * Clean all contents
     *
     * @command inside:clean-all-content
     * @usage inside:clean-all-content
     * @return void
     */
    public function clean_content($options = []): void
    {
        new \Inside\Host\Services\InsideService;

        $models = \Inside\Content\Models\Model::all();

        foreach ($models as $model) {
            if ($model->class === 'Inside\Content\Models\Contents\Users') {
                continue;
            }

            call_user_func($model->class . '::unsetEventDispatcher');
        }

        Inside::withoutDrupalModule(function () {
            $entityTypes = [
                [
                    'type' => 'files',
                    'class' => '\Drupal\file\Entity\File',
                ],
                [
                    'type' => 'paragraphs',
                    'class' => '\Drupal\paragraphs\Entity\Paragraph',
                ],
                [
                    'type' => 'nodes',
                    'class' => '\Drupal\node\Entity\Node',
                ],
                [
                    'type' => 'comments',
                    'class' => '\Drupal\comment\Entity\Comment',
                ],
                [
                    'type' => 'menu links',
                    'class' => '\Drupal\menu_item_extras\Entity\MenuItemExtrasMenuLinkContent',
                ],
            ];

            foreach ($entityTypes as $entityType) {
                $this->output()->writeln(sprintf('Cleaning %s', $entityType['type']));

                $entities = call_user_func($entityType['class'] . '::loadMultiple');

                if (empty($entities)) {
                    $this->output()->writeln(sprintf('No %s found, skipping', $entityType['type']));
                    continue;
                }

                foreach ($entities as $entity) {
                    $entity->delete();
                }

                $this->output()->writeln(sprintf("All %s have been deleted", $entityType['type']));
            }
        });

        $this->clean_users();

        foreach ($models as $model) {
            $model = new $model->class;

            if ($model->getTable() === 'inside_content_users') {
                continue;
            }

            \DB::table($model->getTable())->truncate();
        }
    }

    /**
     * Clean unsync contents
     *
     * @command inside:clean
     * @usage inside:clean
     * @return void
     */
    public function clean($options = []): void
    {
        new \Inside\Host\Services\InsideService;

        $models = \Inside\Content\Models\Model::all();

        foreach ($models as $model) {
            $items = call_user_func($model->class . '::all');
            $entityType = guess_drupal_entity_type($model->class);
            $class = guess_drupal_entity_class($entityType);

            if ($entityType) {
                foreach ($items as $item) {
                    $entity = \Drupal::service('entity.repository')->loadEntityByUuid($entityType, $item->uuid_host);

                    if (!$entity) {
                        $this->output()->writeln('Delete ' . $model->class . ' ' . $item->uuid_host);
                        $item = call_user_func($model->class . '::where', ['uuid_host' => $item->uuid_host]);
                        $item->delete();
                    }
                }
            }
        }

        $this->output()->writeln('Useless contents have been removed.');
    }

    /**
     * Sync contents
     *
     * @command inside:sync
     * @usage inside:sync
     * @return void
     */
    public function sync($options = []): void
    {
        new \Inside\Host\Services\InsideService;

        $models = \Inside\Content\Models\Model::all();
        $explored = [];

        $bridge = new BridgeContent();

        // Disable events
        foreach ($models as $model) {
            call_user_func($model->class . '::unsetEventDispatcher');
        }

        foreach ($models as $model) {
            $entityType = guess_drupal_entity_type($model->class);
            $class = guess_drupal_entity_class($entityType);

            if ($class && !in_array($class, $explored)) {
                $entities = call_user_func($class . '::loadMultiple');

                foreach ($entities as $entity) {
                    if ($entity && get_class($entity) != 'Drupal\paragraphs\Entity\Paragraph' && $entity->uuid()) {
                        // Skip anonymous user
                        if ($entityType === 'user' && $entity->id() == 0) {
                            continue;
                        }

                        try {
                            $this->output()->writeln('Sync ' . $entity->bundle() . ' from host ' . $entity->uuid());

                            $serializer = \Drupal::service('serializer');
                            $normalized = $serializer->normalize($entity);

                            // Paragraphs translations created in back office need to be resaved with the bridge
                            $paragraphs = $this->extractParagraphs($entity);

                            foreach ($entity->getTranslationLanguages() as $langcode => $language) {
                                $translation = $entity->getTranslation($langcode);

                                $translatedParagraphs = $this->extractParagraphs($translation);

                                $datas = [
                                    'uuid_host' => $entity->uuid(),
                                    'author' => $normalized['author'],
                                    'created_at' => $normalized['created_at'],
                                    'content' => !empty($translatedParagraphs) ? $translatedParagraphs : $paragraphs,
                                    'langcode' => $langcode,
                                    'type' => 'node',
                                    'bundle' => $normalized['bundle'],
                                ];

                                $bridge->contentUpdate($translation->bundle(), $datas);

                                $translation->save();
                            }
                        } catch (\Exception $e) {
                            //
                        }
                    }
                }

                $explored[] = $class;
            }
        }

        $this->output()->writeln('Contents have been synced.');
    }

    /**
     * Extract paragraphs from an entity
     *
     * @param Entity $entity
     * @return array
     */
    private function extractParagraphs(Entity $entity)
    {
        $serializer = \Drupal::service('serializer');
        $normalized = $serializer->normalize($entity);

        $paragraphs = [];

        if (isset($normalized['content']) && is_array($normalized['content'])) {
            $entityParagraphs = $entity->get('field_content')->getValue();

            foreach ($entityParagraphs as $paragraph) {
                $paragraph = Paragraph::load($paragraph['target_id']);

                if ($paragraph->hasTranslation($entity->language()->getId())) {
                    $paragraph = $paragraph->getTranslation($entity->language()->getId());
                }

                $normalizedParagraph = $serializer->normalize($paragraph);

                $fields = ['sectionable_uuid', 'sectionable_type', 'field', 'uuid_host', 'created_at', 'status'];

                foreach ($fields as $remove) {
                    unset($normalizedParagraph[$remove]);
                }

                $normalizedParagraph['langcode'] = $entity->language()->getId();

                if (isset($normalizedParagraph['content'])) {
                    $normalizedParagraph['content'] = $this->extractParagraphs($paragraph, true);
                }

                $paragraphs[] = $normalizedParagraph;
            }
        }

        return $paragraphs;
    }

    /**
     * Remove all but maecia users
     *
     * @command inside:remove-users
     * @usage inside:remove-users
     * @return void
     */
    public function clean_users($options = []): void
    {
        new \Inside\Host\Services\InsideService;

        $this->output()->writeln('Cleaning users');
        $items = Users::all();

        $bridge = new BridgeContent();

        foreach ($items as $item) {
            if (strstr($item->email, 'maecia.com')) {
                continue;
            }

            $bridge->contentDelete('users', $item->uuid);
        }

        $this->output()->writeln('Users have been removed.');
    }

    /**
     * Resave all or a specific user(s)
     *
     * @command inside:save-users
     * @usage inside:save-users [--mail MAIL]
     * @option mail A specfic mail address
     * @return void
     */
    public function save_users($options = ['mail' => '']): void
    {
        $mail = $this->input()->hasOption('mail') ? $this->input()->getOption('mail') : false;

        if (!empty($mail)) {
            $users = [user_load_by_mail($mail)];
        } else {
            $users = User::loadMultiple();
        }

        /** @var User $user */
        foreach ($users as $user) {
            if ($user->id() === '0') {
                continue;
            }
            $user->save();
        }

        $this->output()->writeln('All user accounts have been saved.');
    }

    /**
     * Export custom configuration.
     *
     * @command inside:export-config
     * @aliases exco
     * @usage inside:export-config
     * @return void
     */
    public function export_custom_config($options = ['name' => 'all']): void
    {
        $global_conf_path = getcwd() . '/vendor/maecia/inside-drupal/config/sync';
        $global_conf = scandir($global_conf_path);
        $module_path = drupal_get_path('module', 'inside');

        // Export the current configuration.
        drush_invoke_process('@self', 'config-export', array('-y', '--destination=' . $global_conf_path));
        $new_global_conf = scandir($global_conf_path);

        // List of changes since last export.
        $new_conf = array_diff($new_global_conf, $global_conf);
        $new_config_path = $module_path . '/config/import/' . $options['name'];

        if (!empty($new_conf)) {
            if (!file_exists($new_config_path)) {
                mkdir($new_config_path, 0777, true);
            }

            // Create a folder with new configurations.
            foreach ($new_conf as $element) {
                copy($global_conf_path . '/' . $element, $new_config_path . '/' . $element);
            }
        }
    }

    /**
     * Importt custom configuration.
     *
     * @command inside:import-config
     * @aliases imco
     * @usage inside:import-config
     * @return void
     */
    public function import_custom_config($options = ['name' => 'all', 'preview' => 'list']): void
    {
        $module_path = drupal_get_path('module', 'inside');
        $new_conf_path = $module_path . '/config/import/' . $options['name'];

        $import_options = array(
            '--partial',
            '--source=' . $new_conf_path
        );

        // Preview mode.
        if ($options['preview'] !== 'list') {
            $import_options[] = '--preview';
            $import_options[] = '--no';
        } else {
            $import_options[] = '--yes';
        }

        drush_invoke_process('@self', 'config-import', $import_options);

        // Resave form display.
        if ($options['preview'] === 'list') {
            drush_invoke_process('@self', 'inside:resave-form-displays');
        }
    }

    /**
     * Create fields based on YAML files.
     *
     * @command inside:init-fields
     * @aliases iifs
     * @usage inside:init-fields --file="path/to/folder"
     *
     * @see https://gist.github.com/jacerider/340edaefa071e4fbf11f7bbba2a9d1f5
     * @throws EntityStorageException
     */
    public function init_fields($file = 'init/fields.yml')
    {
        $random = new Random();
        $messenger = \Drupal::messenger();
        $entity_manager = \Drupal::service('entity.repository');
        $language_manager = \Drupal::languageManager();
        $config_factory = \Drupal::service('config.factory');
        $data = Yaml::decode(file_get_contents($file)) ?: [];
        $node_types = $data['node_types'] ?? [];
        $field_storages = $data['field_storages'] ?? [];
        $field_instances = $data['field_instances'] ?? [];

        // Node types.
        foreach ($node_types as $bundle => $node_type_config) {
            $node_type = NodeType::load($bundle);
            if (!$node_type) {
                NodeType::create([
                    'type' => $node_type_config['type'],
                    'name' => $node_type_config['name'],
                ])->save();
                $node_type = NodeType::load($bundle);
                $messenger->addMessage('New node type: ' . $bundle);
            }

            // Update node type attributes;
            foreach ($node_type_config as $key => $value) {
                $node_type->set($key, $value);
            }

            if ($node_type_config['third_party_settings'] ?? NULL) {
                foreach ($node_type_config['third_party_settings'] as $module => $attributes) {
                    foreach ($attributes as $key => $value) {
                        $node_type->setThirdPartySetting($module, $key, $value);
                    }
                }
            }

            // Enable/disable content translation.
            $is_translatable = ($node_type_config['translatable'] ?? FALSE) || !empty($node_type_config['translations']);
            $translation_config = $config_factory->getEditable('language.content_settings.node.' . $bundle);
            $translation_config->set('id', 'node.' . $bundle);
            $translation_config->set('target_entity_type_id', 'node');
            $translation_config->set('target_bundle', $bundle);
            $translation_config->set('default_langcode', 'site_default');
            $translation_config->set('language_alterable', $is_translatable);
            $translation_config->set('third_party_settings', [
                    'content_translation' => [
                        'enabled' => $is_translatable,
                        'bundle_settings' => [
                            'untranslatable_fields_hide' => '0',
                        ],
                    ],
                ] + ($node_type_config['third_party_settings']['content_translation'] ?? []));
            $translation_config->save();

            if ($node_type_config['translations'] ?? NULL) {
                foreach ($node_type_config['translations'] as $langcode => $translations) {
                    $field_config = 'node.type.' . $bundle;
                    foreach ($translations as $key => $value) {
                        $language_manager->getLanguageConfigOverride($langcode, $field_config)->set($key, $value)->save();
                    }
                }
            }

            // Save/Update changes for this Node Type.
            $node_type->save();

            // Log this operation.
            $messenger->addMessage('Node type ' . $bundle . ' updated.');
        }

        // Field storages.
        foreach ($field_storages as $field_name => $field_config) {
            foreach ($field_config['storages'] as $entity_type) {
                $field_storage = FieldStorageConfig::loadByName($entity_type, $field_name);
                if (empty($field_storage)) {
                    $field_storage = FieldStorageConfig::create(array(
                        'field_name' => $field_name,
                        'entity_type' => $entity_type,
                        'type' => $field_config['type'],
                        'langcode' => $field_config['langcode'] ?? 'fr',
                        'cardinality' => $field_config['cardinality'] ?? 1,
                        'translatable' => $field_config['translatable'] ?? 1,
                        'settings' => $field_config['settings'] ?? [],
                    ))->save();
                    // Log this operation.
                    $messenger->addMessage('Field storage created for "' . $field_name . '"');
                }
            }
        }

        // Field instances.
        foreach ($field_instances as $field_name => $field_config) {
            foreach ($field_config['instances'] as $target) {
                [$entity_type, $bundle] = explode(':', $target);

                $config_array = array(
                    'field_name' => $field_name,
                    'entity_type' => $entity_type,
                    'bundle' => $bundle,
                    'label' => $field_config['label'],
                    'required' => $field_config['required'],
                    'langcode' => $field_config['langcode'] ?? 'fr',
                    'settings' => $field_config['settings'] ?? [],
                );

                if ($bundle) {
                    // Load or create field instance.
                    $field_instance = FieldConfig::loadByName($entity_type, $bundle, $field_name);
                    if (empty($field_instance)) {
                        FieldConfig::create($config_array)->save();
                        $field_instance = FieldConfig::loadByName($entity_type, $bundle, $field_name);
                        $messenger->addMessage('Field instance created for : ' . $field_name);
                    }

                    $field_instance->setLabel($field_config['label']);
                    $field_instance->setRequired($field_config['required']);
                    if ($field_config['langcode'] ?? NULL) {
                        $field_instance->set('langcode', $field_config['langcode']);
                    }

                    // Quick fix for Image field.
                    if ($field_instance->getType() == 'image') {
                        // Create file entity.
                        file_put_contents('public://inside-dummy-image.jpg', 'inside');
                        $image = File::create(['uri' => 'public://inside-dummy-image.jpg']);
                        $image->save();
                        if ($uuid = $image->uuid()) {
                            $field_config['settings']['default_image']['uuid'] = $uuid;
                        }
                    }

                    // Always update field instance configuration.
                    $field_instance->set('settings', $field_config['settings']);

                    if ($field_config['widget'] ?? NULL) {
                        // Form modes.
                        foreach ($field_config['widget'] as $form_mode => $widget_config) {
                            \Drupal::service('entity_display.repository')->getFormDisplay($entity_type, $bundle, $form_mode)
                                ->setComponent($field_name, $widget_config)
                                ->save();
                            $messenger->addMessage('Form mode "' . $form_mode . '" updated for "' . $field_name . '"');
                        }
                    }

                    if ($field_config['formatter'] ?? NULL) {
                        // Display modes.
                        foreach ($field_config['formatter'] as $display_mode => $formatter) {
                            $view_modes = $entity_manager->getViewModes($entity_type);
                            if (isset($view_modes[$display_mode]) || $display_mode == 'default') {
                                entity_get_display($entity_type, $bundle, $display_mode)
                                    ->setComponent($field_name, !is_array($formatter) ? $field_config['formatter']['default'] : $formatter)
                                    ->save();
                                $messenger->addMessage('Display mode "' . $display_mode . '" updated for "' . $field_name . '"');
                            }
                        }
                    }
                }

                if ($field_config['translations'] ?? NULL) {
                    if (!$field_instance->isTranslatable()) {
                        // Enable translation if not set yet.
                        $field_instance->setTranslatable(TRUE);
                    }
                    foreach ($field_config['translations'] as $langcode => $translations) {
                        $translatation_field_config = 'field.field.' . $entity_type . '.' . $bundle . '.' . $field_name;
                        foreach ($translations as $key => $value) {
                            $language_manager->getLanguageConfigOverride($langcode, $translatation_field_config)->set($key, $value)->save();
                        }
                    }
                }

                // Save the field now.
                $field_instance->save();

                // Log this operation.
                $messenger->addMessage('Field instance updated for : ' . $field_name);
            }
        }
    }

    /**
     * Create default content.
     *
     * @command inside:create-default-content
     * @alias icdc
     * @usage inside:create-default-content --file="path/to/folder"
     */
    public function create_default_content($file = 'init/content.yml')
    {
        $random = new Random();
        $messenger = \Drupal::messenger();
        $entity_manager = \Drupal::service('entity.repository');
        $language_manager = \Drupal::languageManager();
        $data = Yaml::decode(file_get_contents($file)) ?: [];
        $default_content = $data['default_content'] ?? [];

        // Default content.
        foreach ($default_content as $type => $list) {
            [$entity_type, $bundle] = explode(':', $type);
            $storage = $entity_manager->getStorage($entity_type);

            foreach ($list as $uuid => $values) {
                $existing = $storage->loadByProperties(['uuid' => $uuid]);
                $entity = !empty($existing) ? reset($existing) : NULL;

                if (!$entity) {
                    $entity = $entity_manager->getStorage($entity_type)->create([
                        'uuid' => $uuid,
                        'type' => $bundle,
                        'title' => $values['title'] ?? $random->name(),
                        'langcode' => $values['langcode'] ?? 'fr',
                        'status' => $values['status'] ?? 1,
                    ]);
                }

                foreach ($values as $key => $value) {
                    if ($entity->hasField($key)) {
                        // Preprocess URI to replace UUID by Node ID.
                        if ($entity->getEntityTypeId() == 'menu_link_content' && $key == 'link' && isset($value['uri'])) {
                            [$link_entity_type, $link_entity_uuid] = explode(':', $value['uri']);
                            if ($link_entity_type == 'node' && Uuid::isValid($link_entity_uuid)) {
                                $found_nodes = $entity_manager->getStorage('node')->loadByProperties(['uuid' => $link_entity_uuid]);
                                $link_entity_target = reset($found_nodes) ?? NULL;
                                if ($link_entity_target) {
                                    $value['uri'] = 'internal:/node/' . $link_entity_target->id();
                                } else {
                                    // Unset UUID because it will cause exception in back office.
                                    $value['uri'] = 'route:<front>';
                                }
                            }
                        }

                        $entity->set($key, $value);
                    }
                }

                // Add node translations.
                $has_translations = $values['translations'] ?? FALSE;
                if ($has_translations && !$entity->isTranslatable()) {
                    if (method_exists($entity, 'setTranslatable')) {
                        $entity->setTranslatable(TRUE);
                    }
                }
                if ($has_translations) {
                    foreach ($values['translations'] as $langcode => $translated_fields) {
                        $translated_entity = $entity->hasTranslation($langcode) ? $entity->getTranslation($langcode) : NULL;
                        if (!$translated_entity) {
                            $translated_entity = $entity->addTranslation($langcode);
                        }
                        foreach ($translated_fields as $key => $value) {
                            if ($translated_entity->hasField($key)) {
                                $translated_entity->set($key, $value);
                            }
                        }
                        // Copy original entity's status.
                        if ($translated_entity->hasField('status')) {
                            $translated_entity->set('status', $entity->get('status')->getValue());
                        }
                        if ($translated_entity->hasField('enabled')) {
                            $translated_entity->set('enabled', $entity->get('enabled')->getValue());
                        }
                    }
                }

                // Save our new entity with everything attached (translation...etc).
                try {
                    $entity->save();
                } catch (\Exception $e) {
                    $messenger->addError($e->getMessage());
                }
            }
        }
    }

    /**
     * Helper function to generate random UUID.
     *
     * @command inside:uuid
     * @aliases iuuid
     * @usage inside:uuid 3
     */
    public function uuid($count = 1)
    {
        $uuids = [];
        $service = \Drupal::service('uuid');
        for ($i = 0; $i < $count; $i++) {
            $uuids[] = $service->generate();
        }
        return $uuids;
    }

    /**
     * Resync PostgreSQL file_managed_fid_seq sequence (for imports or migrations).
     *
     * @command inside:resync-file-sequence
     * @aliases resync-fid-seq
     */
    public function resyncFileManagedSequence(): void {
        $connection = \Drupal::database();

        if ($connection->driver() !== 'pgsql') {
            $this->output()->writeln('<error>This command only supports PostgreSQL.</error>');
            return;
        }

        try {
            $max_fid = $connection->query('SELECT MAX(fid) FROM file_managed')->fetchField();
            $next_fid = (int) $max_fid + 1;

            $connection->query("ALTER SEQUENCE file_managed_fid_seq RESTART WITH $next_fid");

            $this->output()->writeln("<info>file_managed_fid_seq successfully reset to $next_fid</info>");
        }
        catch (\Exception $e) {
            $this->output()->writeln('<error>Failed to resync file_managed_fid_seq: ' . $e->getMessage() . '</error>');
        }
    }

    /**
     * Display information about a field (all instances / storage).
     *
     * @command inside:field-info
     * @aliases ifi
     * @usage inside:field-info field_duration
     *
     * @param string $field_name
     *   The machine name of the field to inspect (e.g. field_duration).
     *
     * @return void
     */
    public function field_info($field_name = '')
    {
        if (empty($field_name)) {
            $this->output()->writeln('<error>Usage: inside:field-info <field_name></error>');
            return;
        }

        $found = false;

        /** @var \Drupal\field\Entity\FieldConfig[] $instances */
        $instances = FieldConfig::loadMultiple();
        foreach ($instances as $field) {
            if ($field->getName() !== $field_name) {
                continue;
            }

            $found = true;
            $bundle = $field->getTargetBundle();
            $entity_type = $field->getTargetEntityTypeId();
            $label = $field->label();
            $instance_id = $field->id(); // ex: field.field.node.article.field_duration

            // Storage config
            $storage = FieldStorageConfig::loadByName($entity_type, $field->getName());
            $storage_type = $storage ? $storage->getType() : 'inconnu';
            $storage_id = $storage ? $storage->id() : 'n/a'; // ex: node.field_duration
            $storage_uuid = $storage ? $storage->uuid() : 'n/a';

            $this->output()->writeln(sprintf('Champ : <info>%s</info>', $field_name));
            $this->output()->writeln(sprintf(' - Bundle       : %s', $bundle));
            $this->output()->writeln(sprintf(' - Entité       : %s', $entity_type));
            $this->output()->writeln(sprintf(' - Label        : %s', $label));
            $this->output()->writeln(sprintf(' - Instance ID  : %s', $instance_id));
            $this->output()->writeln(sprintf(' - Storage ID   : %s', $storage_id));
            $this->output()->writeln(sprintf(' - Storage UUID : %s', $storage_uuid));
            $this->output()->writeln(sprintf(' - Storage type : %s', $storage_type));
            $this->output()->writeln(''); // blank line for readability
        }

        if (!$found) {
            $this->output()->writeln(sprintf('<comment>Aucune instance trouvée pour le champ : %s</comment>', $field_name));
        }
    }
}

