<?php

namespace Inside\Host\Bridge\Traits;

use Drupal\Core\Entity\Display\EntityFormDisplayInterface;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\locale\SourceString;
use Drupal\locale\StringStorageException;
use Exception;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;
use Inside\Content\Facades\Schema;

trait ManageFieldGroup
{
    /**
     * Prépare system default form groups
     *
     * @param  EntityFormDisplayInterface  $formDisplay
     * @param  string  $bundle
     *
     * @return bool
     * @throws StringStorageException
     */
    protected function prepareSystemGroups(EntityFormDisplayInterface $formDisplay, string $bundle): bool
    {
        $groups = [
            [
                'group_name' => 'group_options',
                'entity_type' => 'node',
                'bundle' => $bundle,
                'children' => [
                    'status',
                    'field_show_image',
                    'langcode',
                ],
                'weight' => 100,
                'parent_name' => '',
                'format_settings' => [
                    'label' => 'Publication option',
                    'description' => '',
                    'required_fields' => true,
                    'id' => '',
                    'classes' => '',
                ],
                'label' => 'Publication option',
                'format_type' => 'fieldset',
            ],
            [
                'group_name' => 'group_advanced_options',
                'entity_type' => 'node',
                'bundle' => $bundle,
                'children' => [
                    'uid',
                    'created',
                    'field_is_featured',
                    'field_is_slideshow',
                ],
                'weight' => 200,
                'parent_name' => '',
                'format_settings' => [
                    'label' => 'Advanced configuration/option',
                    'description' => '',
                    'required_fields' => true,
                    'id' => '',
                    'classes' => '',
                ],
                'label' => 'Advanced configuration/option',
                'format_type' => 'fieldset',
            ],
        ];

        $groupLabels = [
            'group_options' => [
                'en' => 'Publication option',
                'fr' => 'Options de publication',
                'es' => 'Opciones de publicación',
                'de' => 'Veröffentlichungsoptionen',
                'nl' => 'Publicatieopties',
                'pt' => 'Opções de publicação',
            ],
            'group_advanced_options' => [
                'en' => 'Advanced configuration/option',
                'fr' => 'Options avancées',
                'es' => 'Opciones avanzadas',
                'de' => 'Erweiterte Konfiguration/Option',
                'nl' => 'Geavanceerde configuratie/optie',
                'pt' => 'Configuração/Opção avançada',
            ],
        ];

        $localeStorage = \Drupal::service('locale.storage');

        foreach ($groups as $group) {
            $localeString = $localeStorage->findString(['source' => $group['label']]);
            if ($localeString === null) {
                $localeString = new SourceString;
                $localeString->setString($group['label']);
                $localeString->setStorage($localeStorage);
                $localeString->save();
            }
            foreach ($this->languages as $language) {
                $localeStorage->createTranslation(
                    [
                        'lid' => $localeString->lid,
                        'language' => $language,
                        'translation' => $groupLabels[$group['group_name']][$language] ?? $group['label'],
                    ]
                )->save();
            }
            if ($formDisplay->getThirdPartySetting('field_group', $group['group_name']) === null) {
                field_group_group_save((object) $group, $formDisplay);
            }
        }

        return true;
    }

    /**
     * Create a new group field ( fieldset )
     *
     * @param  string  $type
     * @param  string  $name
     * @param  array  $labels
     * @param  int  $weight
     *
     * @return bool
     * @throws StringStorageException
     */
    public function createOrUpdateFieldGroup(string $type, string $name, array $labels, int $weight = 0): bool
    {
        $entityTypeId = 'node';
        if ($type === 'users') {
            $entityTypeId = 'user';
        }
        $form = \Drupal::service('entity_display.repository')->getFormDisplay($entityTypeId, $type === 'users' ? 'user' : $type);

        // Prepare labels
        $localeStorage = \Drupal::service('locale.storage');
        $localeString = $localeStorage->findString(['source' => $name]);
        if ($localeString === null) {
            $localeString = new SourceString;
            $localeString->setString($name);
            $localeString->setStorage($localeStorage);
            $localeString->save();
        }
        foreach ($this->languages as $language) {
            $localeStorage->createTranslation(
                [
                    'lid' => $localeString->lid,
                    'language' => $language,
                    'translation' => $labels[$language] ?? '',
                ]
            )->save();
        }
        $group = [
            'group_name' => $name,
            'entity_type' => 'node',
            'bundle' => $type,
            'children' => [],
            'weight' => $weight,
            'parent_name' => '',
            'format_settings' => [
                'label' => Arr::first($labels),
                'description' => '',
                'required_fields' => true,
                'id' => '',
                'classes' => '',
            ],
            'label' => Arr::first($labels),
            'format_type' => 'fieldset',
        ];
        if (($groupInfos = $form->getThirdPartySetting('field_group', $name)) !== null) {
            $group = array_merge($groupInfos, $group);
        }

        return field_group_group_save((object) $group, $form) !== null;
    }

    /**
     * Detach field $fieldName from any group
     *
     * @param  EntityFormDisplayInterface  $formDisplay
     * @param  string  $fieldName
     *
     * @throws EntityStorageException
     */
    protected function detachFieldFromAnyGroup(EntityFormDisplayInterface $formDisplay, string $fieldName)
    {
        foreach ([
                     'group_options',
                     'group_advanced_options',
                 ] as $group
        ) {
            $groupInfos = $formDisplay->getThirdPartySetting('field_group', $group, []);
            $drupalField = array_flip($this->swappableNames);
            if (array_key_exists($fieldName, $drupalField)) {
                $fieldName = $drupalField[$fieldName];
            } else {
                $fieldName = $this->getDrupalFieldNameFromInsideFieldName($fieldName);
            }
            if (array_key_exists('children', $groupInfos)
                && ($key = array_search($fieldName, $groupInfos['children'])) !== false
            ) {
                unset($groupInfos['children'][$key]);
                $formDisplay->setThirdPartySetting('field_group', $group, $groupInfos);
                $formDisplay->save();
            }
        }
    }

    /**
     * Attach a widget to a group
     *
     * @param  EntityFormDisplayInterface  $formDisplay  form display
     * @param  array  $options  widget option
     * @param  string  $type  content type
     * @param  string  $fieldName  field name
     *
     * @return bool
     * @throws Exception
     */
    protected function attachFieldToGroup(
        EntityFormDisplayInterface $formDisplay,
        array $options,
        string $type,
        string $fieldName
    ): bool {
        if (!array_key_exists('group', $options['widget'])) {
            Log::debug('[BridgeContentType::attachToGroup] widget group is not set');

            return false;
        } else {
            $groupInfos = $formDisplay->getThirdPartySetting('field_group', $options['widget']['group']);

            Schema::refresh();
            if ($groupInfos === null) {
                // Group does not exist
                Log::error(
                    '[BridgeContentType::attachToGroup] group ('.$options['widget']['group'].') does not exists'
                );

                return false;
            }
            if (!array_key_exists('children', $groupInfos)) {
                $groupInfos['children'] = [];
            } else {
                // Clean children if needed
                foreach ($groupInfos['children'] as $key => $childFieldName) {
                    $childFieldName = $this->getInsideFieldNameFromDrupalFieldName(
                        $this->swappableNames[$childFieldName] ?? $childFieldName
                    );
                    if (!Schema::hasField($type, $childFieldName)) {
                        unset($groupInfos['children'][$key]);
                    }
                }
            }
            $this->detachFieldFromAnyGroup($formDisplay, $fieldName);

            $drupalField = array_flip($this->swappableNames);
            if (array_key_exists($fieldName, $drupalField)) {
                $fieldName = $drupalField[$fieldName];
            } elseif (!in_array($fieldName, $this->drupalFields['node'])) {
                $fieldName = $this->getDrupalFieldNameFromInsideFieldName($fieldName);
            }
            if (!in_array($fieldName, $groupInfos['children'])) {
                $groupInfos['children'][] = $fieldName;
            }

            // Rename drupal fields
            foreach ($groupInfos['children'] as $key => $childFieldName) {
                if (array_key_exists($childFieldName, $drupalField)) {
                    $groupInfos['children'][$key] = $drupalField[$childFieldName];
                }
            }

            $groupInfos['children'] = array_values($groupInfos['children']);
            try {
                unset($groupInfos['group_name'], $groupInfos['entity_type'], $groupInfos['bundle'], $groupInfos['mode'], $groupInfos['form'], $groupInfos['context']);
                $formDisplay->setThirdPartySetting('field_group', $options['widget']['group'], $groupInfos);
                $formDisplay->save();
            } catch (Exception $e) {
                return false;
            }

            return true;
        }
    }

    /**
     * get Form Field Group options
     *
     * @param  string  $type
     * @param  string  $name
     * @return array|null
     */
    public function getFormGroupOptions(string $type, string $name): ?array
    {
        $formDisplay = \Drupal::service('entity_display.repository')->getFormDisplay(
            $type == 'users' ? 'user' : 'node',
            $type == 'users' ? 'user' : $type
        );

        $settings = $formDisplay->getThirdPartySetting('field_group', $name);
        if (!$settings) {
            return null;
        }
        // Convert drupal field name to inside ones
        foreach ($settings['children'] as &$children) {
            $children = str_replace('field_', '', $children);
        }

        return $settings;
    }
}
