<?php
declare(strict_types=1);

namespace Inside\Host\Bridge;

use Auth;
use DB;
use Drupal;
use Drupal\Core\Entity\EntityInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\Core\Field\EntityReferenceFieldItemList;
use Drupal\comment\Entity\Comment;
use Drupal\node\Entity\Node;
use Drupal\paragraphs\Entity\Paragraph;
use Exception;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Schema;
use Inside\Authentication\Models\User;
use Inside\Content\Events\ContentFullyInsertedEvent;
use Inside\Content\Events\ContentFullyUpdatedEvent;
use Inside\Content\Events\ContentSavedWithImages;
use Inside\Content\Models\Content;
use Inside\Content\Models\InsideLog;
use Inside\Content\Models\Section;
use Inside\Permission\Facades\Permission;
use Inside\Search\Facades\Searchable;
use Log;

/**
 * Bridge for content creation
 *
 * @category  Class
 * @package   Inside\Host\Bridge\BridgeContent
 * @link      http://www.maecia.com/
 */
final class BridgeContent extends Bridge
{
    /**
     * The entity drupal type
     */
    protected string $type = 'node';

    /**
     * The entity drupal class
     */
    protected string $class = Node::class;

    /**
     * @var string Default lowest log level
     */
    protected string $lowestLogLevel = 'debug';


    public function __construct(bool $fullLoad = true)
    {
        parent::__construct($fullLoad);
        $this->lowestLogLevel = env('LOWEST_LOG_LEVEL', 'debug');
    }

    protected function extractParagraphsFromDatas(
        string $domain,
        string $type,
        array &$datas,
        Collection $paragraphs
    ): Collection {
        $fields = collect(get_drupal_paragraph_fields($domain, $type))->transform(
            function ($field) {
                return str_replace('field_', '', $field);
            }
        );

        $fields->each(
            function ($field) use (&$datas, &$paragraphs) {
                if (isset($datas[$field]) && is_array($datas[$field])) {
                    $paragraphs[$field] = $datas[$field]; // Save this for later
                    if (! empty($paragraphs[$field])) {
                        // IMPORTANT NOTE : ici, je n'efface le champ que si notre contenu
                        // a des paragraphes pour le cas ou nous sommes en édition et que
                        // le contenu a déjà un ou plusieurs paragraphes, il faut de se
                        // fait, le ou les effacer.
                        unset($datas[$field]); // We must save our content first
                    }
                }
            }
        );

        return $fields;
    }

    /**
     * Prepare data before insertion
     */
    protected function prepareData(string $type, array &$data, bool $creation = true): void
    {
        if (! isset($data['type'])) {
            if ($type === 'users') {
                $data['type'] = 'user';
            } else {
                $data['type'] = 'node';
            }
        }
        if (! isset($data['bundle'])) {
            if ($type === 'users') {
                $data['bundle'] = 'user';
            } else {
                $data['bundle'] = $type;
            }
        }
        if ($creation && ! isset($data['author'])) {
            $superAdmin = User::where('email', config('app.technical_mail'))->first()->uuid;
            if (! App::runningInConsole() && isset($data['authors'])) {
                $data['author'] = $data['authors'];
            } else {
                $data['author'] = $superAdmin;
            }
        }
        unset($data['authors']);

        if ($creation) {
            if (! isset($data['created_at'])) {
                $data['created_at'] = now()->format('Y-m-d H:i:s');
            }
            if (! isset($data['published_at'])) {
                $data['published_at'] = now()->format('Y-m-d H:i:s');
            }
        } else {
            if (! isset($data['updated_at'])) {
                $data['updated_at'] = now()->format('Y-m-d H:i:s');
            }
            if (isset($data['created_at'])) {
                unset($data['created_at']);
            }
        }

        if (! $creation && isset($data['uuid']) && ! isset($data['langcode'])) {
            $content = $this->getContent($type, $data['uuid']);
            if (! is_null($content)) {
                $data['langcode'] = $content->langcode;
            }
        }
    }

    protected function getContent(string $type, string $uuid): Content | Section | null
    {
        return Permission::withoutAllowedScope(
            function () use ($type, $uuid) {
                return call_user_func(
                    ($this->type === 'paragraph' ? section_type_to_class($type) : type_to_class($type))
                    .'::find',
                    $uuid
                );
            }
        );
    }

    /**
     * Create content
     * @throws Exception
     */
    public function contentInsert(
        string $type,
        array $data,
        bool $creation = true,
        bool $fromCli = false,
        bool $ignoreParagraphs = false
    ): ?string
    {
        if ($creation && isset($data['uuid']) & ! empty($data['uuid'])) {
            // This is not a creation!!
            $creation = false;
        }
        $lock = Drupal::lock();
        $lockKey = ($creation ? 'create_' : 'update_').$type;
        Log::{$this->lowestLogLevel}('[BridgeContent:contentInsert', ['lockKey' => $lockKey]);
        $limitLoop = 0;
        while (! $lock->acquire($lockKey)) {
            if ($limitLoop > 100) {
                $lock->release($lockKey);
                throw new Exception("Possible infinite loop while trying to lock content");
            }
            $lock->wait($lockKey);
            $limitLoop++;
        }
        Log::{$this->lowestLogLevel}('[BridgeContent:contentInsert -> prepareData');
        $this->prepareData($type, $data, $creation);

        $onlyInsideUpdates = $this->extractOnlyInsideFieldsFromData($type, $data, $creation);
        Log::{$this->lowestLogLevel}('[BridgeContent:contentInsert -> extractOnlyInsideFieldsFromData');
        // Extract paragraph information from $datas for this $type
        $paragraphs = collect();
        $paragraphFields = $ignoreParagraphs ? collect() :
            $this->extractParagraphsFromDatas('node', $type, $data, $paragraphs);

        // check if content exists
        $exists = $this->contentExists($type, $data);
        Log::{$this->lowestLogLevel}('[BridgeContent:contentInsert -> contentExists');
        // Check we have some paragraphFields
        if (!$ignoreParagraphs && $paragraphs->some(fn ($paragraph) => ! empty($paragraph))) {
            Log::{$this->lowestLogLevel}('[BridgeContent:contentInsert -> contentInsertWithParagraphs', ['paragraphs' => $paragraphs]);
            $contentUuid = $this->contentInsertWithParagraphs($type, $data, $paragraphs, $paragraphFields);
            if (! empty($onlyInsideUpdates)) {
                DB::table(type_to_table($type))->where('uuid', $contentUuid)->update($onlyInsideUpdates);
            }
            $this->fireInsertedEvent(
                $type,
                $contentUuid,
                $data,
                $exists
            );

            $lock->release($lockKey);
            return $contentUuid;
        }

        $contentUuid = $this->contentInsertWithoutParagraphs($type, $data);

        Log::{$this->lowestLogLevel}('[BridgeContent:contentInsert -> contentInsertWithoutParagraphs');
        if (! empty($onlyInsideUpdates)) {
            DB::table(type_to_table($type))->where('uuid', $contentUuid)->update($onlyInsideUpdates);
        }
        if (! is_null($contentUuid)) {
            $this->fireInsertedEvent(
                $type,
                $contentUuid,
                $data,
                $exists
            );
        }
        $lock->release($lockKey);
        Log::{$this->lowestLogLevel}('[BridgeContent:contentInsert -> lock release');
        if ($fromCli) {
            $query = call_user_func(type_to_class($type).'::withoutGlobalScopes');
            $content = $query->find($contentUuid);
            if (! is_null($content)) {
                ContentSavedWithImages::dispatch($content);
            }
        }

        return $contentUuid;
    }

    /**
     * Check if a content exists with given data.
     */
    public function contentExists(string $type, array $data): bool
    {
        if (! array_key_exists('uuid', $data) && ! array_key_exists('uuid_host', $data)) {
            return false;
        }
        $table = type_to_table($type);

        $content = call_user_func(type_to_class($type).'::query');

        if (array_key_exists('uuid', $data)) {
            $content->where($table.'.uuid', $data['uuid']);
        } else {
            $content->where($table.'.uuid_host', $data['uuid_host']);
        }

        if ($type !== 'users') {
            if (array_key_exists('langcode', $data)) {
                $langCode = $data['langcode'];
            } else {
                $langCode = Drupal::languageManager()->getCurrentLanguage()->getId();
            }

            $content->where($table.'.langcode', $langCode);
        }

        return $content->exists();
    }

    /**
     * Fire an event when the content has been fully processed
     */
    public function fireInsertedEvent($type, $uuid, $data, bool $updated = false): void
    {
        // Fire a new event when the sections have been attached (the ContentInsertedEvent is fired before attaching sections)
        $entity = $this->getEntity($type, $uuid);
        if ($entity === null) {
            Log::error('fireInsertedEvent could not get entity type ['.$type.'] uuid <'.$uuid.'>');

            return;
        }
        $langCode = $data['langcode'] ?? env('APP_LOCALE', 'en');
        $model = get_lumen_entity($entity, $langCode);

        if ($model instanceof Content) {
            if (! $updated) {
                ContentFullyInsertedEvent::dispatch($model, Auth::user());
            } else {
                ContentFullyUpdatedEvent::dispatch($model, Auth::user());
            }
        }
    }

    /**
     * Simple Insert content with no paragraph
     */
    protected function contentInsertWithoutParagraphs(string $type, array $datas): ?string
    {
        $this->type = guess_drupal_entity_type($type);
        $this->class = guess_drupal_entity_class($this->type);

        $datas = $this->contentAddUuidHostIfPossible('contents', $type, $datas);

        Log::{$this->lowestLogLevel}(
            __(
                '[BridgeContent::contentInsertWithoutParagraphs]'
            ),
            [
                'data' => $datas,
            ]
        );

        $entity = $this->unserializeData($datas);

        $entity->save();

        return get_lumen_entity_uuid($entity);
    }

    /**
     * Content insert that contains at least one paragraph field
     * @throws Exception
     */
    protected function contentInsertWithParagraphs(
        string $type,
        array $datas,
        Collection $paragraphs,
        Collection $paragraphFields
    ): string {
        $this->type = guess_drupal_entity_type($type);
        $this->class = guess_drupal_entity_class($this->type);

        $domain = $this->type === 'node' ? 'contents' : 'sections';

        $datas = $this->contentAddUuidHostIfPossible($domain, $type, $datas);

        if (! isset($datas['langcode'])) {
            $datas['langcode'] = Drupal::languageManager()->getCurrentLanguage()->getId();
        }

        Log::{$this->lowestLogLevel}(
            __(
                '[BridgeContent::contentInsertWithParagraphs]'
            ),
            $datas
        );

        // We first save here without paragraphs but keep entity for later use
        $serializer = Drupal::service('serializer');
        $entity = $serializer->denormalize($datas, $this->class);
        $entity->save();

        // Get corresponding Inside content uuid
        $contentUuid = get_lumen_entity_uuid($entity);
        // We have some paragraph, we must save them
        // And reference them
        $paraToSave = [];
        $subParagraphDatas = [];
        foreach ($paragraphs as $field => $fieldParagraphs) {
            foreach ($fieldParagraphs as $paragraph) {
                $subParagraph = false;

                $bundle = $paragraph['bundle'];
                unset($paragraph['bundle']);
                $paragraphData = array_merge(
                    [
                        "langcode" => $datas['langcode'],
                        "author" => $datas['author'],
                        "status" => true,
                        "created_at" => $datas['created_at'],
                        "sectionable_uuid" => $contentUuid,
                        "field" => $field,
                        "bundle" => $bundle,
                        "sectionable_type" => Str::studly($type),
                    ],
                    $paragraph
                );

                if (
                    array_key_exists('pgID', $paragraphData)
                    && is_string($paragraphData['pgID'])
                    && preg_match('/^[a-f\d]{8}(-[a-f\d]{4}){4}[a-f\d]{8}$/i', $paragraphData['pgID'])
                    && ! array_key_exists('uuid', $paragraphData)
                ) {
                    $paragraphData['uuid'] = $paragraphData['pgID'];
                }

                // TODO check uuid seems to be a uuid_host ( that is wrong )
                $paragraphData = $this->contentAddUuidHostIfPossible('sections', $bundle, $paragraphData);

                // Handle sub paragraphs
                if (isset($paragraphData['content'])) {
                    $subParagraph = $paragraphData;
                    unset($paragraphData['content']);
                }

                // Save our paragraph
                $paragraph = $serializer->denormalize($paragraphData, Paragraph::class);
                $paragraph->save();

                // Add our paragraph to be linked later in our drupal content
                $paraToSave[$field][] = [
                    'target_id' => $paragraph->id(),
                    'target_revision_id' => $paragraph->getRevisionId(),
                ];

                if ($subParagraph) {
                    $subParagraphDatas[$paragraph->id()][] = $subParagraph;
                }
            }
        }

        // reload the entity in case it has changed in the meantime
        $entity = $this->getEntity($type, $contentUuid);

        if ($entity) {
            // Set field_content
            $paragraphFields->each(
                function ($field) use ($entity, $paraToSave) {
                    $entity->{"field_$field"} = $paraToSave[$field];
                }
            );

            // avoid weird duplicates in the cachetags table
            DB::table('cachetags')->where('tag', $entity->getEntityTypeId().':'.$entity->id())->delete();
            $entity->save();
        }

        // Handle sub paragraphs
        // TODO rework this
        if (! empty($subParagraphDatas)) {
            foreach ($subParagraphDatas as $paragraphID => $paragraphArray) {
                $entity = Paragraph::load($paragraphID);
                $entity = $serializer->normalize($entity);
                unset($entity['uuid']);

                foreach ($paragraphArray as $paragraphData) {
                    // Extract paragraph information from $datas for this $type
                    $subParagraphs = collect();
                    $subParagraphsFields = $this->extractParagraphsFromDatas(
                        'paragraph',
                        $paragraphData['bundle'],
                        $paragraphData,
                        $subParagraphs
                    );

                    // Check we have some paragraphFields
                    if ($subParagraphs->some(fn ($subparagraph) => ! empty($subparagraph))) {
                        $this->contentInsertWithParagraphs(
                            $paragraphData['bundle'],
                            Arr::except($entity, ['content']),
                            $subParagraphs,
                            $subParagraphsFields
                        );
                    }
                }
            }
        }

        return $contentUuid;
    }

    /**
     * Add uuid_host if possible
     */
    protected function contentAddUuidHostIfPossible(string $domain, string $type, array $data): array
    {
        // If update we need to set uuid_host
        if (isset($data['uuid_host'])) {
            return $data; // already set
        }
        if (isset($data['uuid'])) {
            try {
                $query = call_user_func(
                    'Inside\\Content\\Models\\'.Str::studly($domain).'\\'.Str::studly($type).'::query'
                );
                // Load content where the comment is to be attached
                $content = $query->findOrFail($data['uuid']);
                $data['uuid_host'] = $content->uuid_host;
            } catch (Exception) {
                Log::error(
                    "[BridgeContent::contentAddUuidHostIfPossible] failed to load Content [$type] [".$data['uuid']."]"
                );
            }
        }

        return $data;
    }

    /**
     * Update content
     * @throws Exception
     */
    public function contentUpdate(
        string $type,
        array $data,
        bool $fromCli = false,
        bool $ignoreParagraphs = false
    ): ?string
    {
        if (isset($data['bundle']) && $data['bundle'] === 'tools' && $data['status'] === false) {
            $this->checkDefaultQuickAccess([$data['uuid_host']], $data['bundle']);
        }

        return $this->contentInsert($type, $data, false, $fromCli, $ignoreParagraphs);
    }

    public function setReferenceDefaultValue(string $type, string $fieldName, array $default): void
    {
        $fieldConfig = FieldConfig::loadByName('node', $type , 'field_'.$fieldName);

        $defaultValues = collect($default)->map( fn($value) => ['target_uuid' => $value])->values()->toArray();
        $fieldConfig?->set('default_value', $defaultValues);
        $fieldConfig?->save();
    }

    public function checkDefaultQuickAccess(array $uuidsToRemove, string $type='tools'): void
    {
        if($type !== 'tools') {
            return;
        }

        $fieldConfig = FieldConfig::loadByName('user', 'user', 'field_tools');
        if (!$fieldConfig) {
            return;
        }

        $defaultValues = collect($fieldConfig->get('default_value'));
        if ($defaultValues->isEmpty()) {
            return;
        }

        $filteredValues = $defaultValues->reject(
            fn ($value) => isset($value['target_uuid']) && in_array($value['target_uuid'], $uuidsToRemove)
        );

        if ($filteredValues->count() !== $defaultValues->count()) {
            $fieldConfig->set('default_value', $filteredValues->values()->toArray());
            $fieldConfig->save();
        }
    }

    /**
     * Delete content entity when inserted from Inside core
     */
    public function contentDelete(string $type, string $uuid): bool
    {
        $this->type = guess_drupal_entity_type($type);

        // Uuid is inside uuid, let's get our uuid_host
        try {
            $content = call_user_func(type_to_class($type).'::findOrFail', $uuid);

            $entity = Drupal::service('entity.repository')->loadEntityByUuid($this->type, $content->uuid_host);

            $userUuid = Auth::user()->uuid ?? 'cli-system';
            if (Schema::hasTable('inside_logs')) {
                InsideLog::create([
                    'user_uuid' => $userUuid,
                    'content_type' => type_to_class($type),
                    'content_uuid' => $uuid,
                    'action' => InsideLog::ACTION_DELETE,
                ]);
            }

            if ($entity) {
                $entity->delete();
                if($entity->getEntityTypeId() === 'comment') {
                    return $content->delete();
                }
                $this->checkDefaultQuickAccess([$content->uuid_host], $type);
                return true;
            } else {
                return $content->delete();
            }
        } catch (Exception $e) {
            Log::error($e->getMessage());
        }

        return false;
    }

    public function getEntity(string $type, string $uuid): ?EntityInterface
    {
        $this->type = guess_drupal_entity_type($type);

        try {
            $content = Permission::withoutAllowedScope(
                function () use ($type, $uuid) {
                    return call_user_func(
                        ($this->type === 'paragraph' ? section_type_to_class($type) : type_to_class($type))
                        .'::findOrFail',
                        $uuid
                    );
                }
            );
            if ($content === null) {
                return null;
            }

            $entity = Drupal::service('entity.repository')->loadEntityByUuid($this->type, $content->uuid_host);

            if ($entity) {
                if ($entity->language()->getId() != $content->langcode && $entity->isTranslatable()) {
                    // We got drupal entity corresponding to our uuid_host but not in the correct language
                    // Let's get a translation of that identity
                    if (! $entity->hasTranslation($content->langcode)) {
                        $entity = $entity->addTranslation($content->langcode, $entity->toArray());
                    } else {
                        $entity = $entity->getTranslation($content->langcode);
                    }
                }

                return $entity;
            }
        } catch (Exception $exception) {
            Log::error('[BridgeContent::getEntity] failed to get entity => '.$exception->getMessage());
        }

        return null;
    }

    protected function unserializeData(array $datas): mixed
    {
        $serializer = Drupal::service('serializer');

        $entity = $serializer->denormalize($datas, $this->class);

        // Have to utf8-encode emojis in in comment's subjects
        return $this->cleanEntity($entity);
    }

    /**
     * Set a content status in database (without triggering events)
     * @throws Exception
     */
    public function setStatus($content, $status)
    {
        $class = get_class($content);
        $contentType = class_to_type($class);

        $node = DB::table('node')->where('uuid', $content->uuid_host)->first();

        if (! $node) {
            return;
        }

        DB::update(
            'update node_field_data set status = '.(int) $status.' where nid = ? and langcode = ?',
            [$node->nid, $content->langcode]
        );
        DB::update(
            'update node_field_revision set status = '.(int) $status.' where nid = ? and langcode = ?',
            [$node->nid, $content->langcode]
        );
        DB::update(
            'update inside_content_'.$contentType.' set status = '.(int) $status.' where uuid = ?',
            [$content->uuid]
        );

        // Index the content if it is searchable
        $searchableClasses = Searchable::getSearchableClasses();

        $allowedTypes = $searchableClasses->filter(
            function ($definition) {
                return Str::startsWith($definition->model, "Inside\\Content\\Models\\Contents\\");
            }
        )->keys();

        if ($allowedTypes->contains($contentType)) {
            $content->searchable();
        }
    }

    protected function extractOnlyInsideFieldsFromData(string $type, array &$data, bool $creation = false): array
    {
        $insideFields = [];

        if ($creation) {
            $insideFields['author_id'] = $data['author'];
        }

        if (in_array($type, ['comments', 'users'])) {
            // Should not have publised_at data ! remove it
            unset($data['published_at']);
        }

        if (array_key_exists('published_at', $data)) {
            $insideFields['published_at'] = $data['published_at'];
            unset($data['published_at']);
        }

        if (array_key_exists('updated_at', $data)) {
            $insideFields['updated_at'] = $data['updated_at'];
            unset($data['updated_at']);
        }

        if (array_key_exists('update_author', $data)) {
            $insideFields['update_author'] = $data['update_author'];
            unset($data['update_author']);
        }

        return $insideFields;
    }

    /**
     * Met à jour tous les champs de type référence d'un contenu Drupal.
     *
     * @param string $type
     * @param string $contentUuidHost
     * @param array $data
     */
    public function updateReferenceFields(string $type, string $contentUuidHost, array $data)
    {
        if($type == 'users') {
            $type = 'user';
        }

        /**
         * @var $entity Node|User
         */
        $entity = Drupal::service('entity.repository')->loadEntityByUuid($type, $contentUuidHost);

        if (!$entity) {
            return;
        }

        foreach ($data as $fieldName => $values) {
            $drupalFieldName = 'field_'.$fieldName;
            if ($entity->hasField($drupalFieldName) && $entity->get($drupalFieldName) instanceof EntityReferenceFieldItemList) {
                $references = [];

                foreach ($values as $uuid) {
                    $referencedEntity = \Drupal::entityTypeManager()
                        ->getStorage($this->getReferencedEntityType($fieldName))
                        ->loadByProperties(['uuid' => $uuid]);
                    if ($referencedEntity) {
                        $referencedEntity = reset($referencedEntity);
                        $references[] = ['target_id' => $referencedEntity->id()];
                    }
                }

                $entity->set($drupalFieldName, $references);
            }
        }

        $entity->save();
    }

    /**
     * Récupère le type d'entité référencé par un champ de type référence.
     *
     * @param string $fieldName
     * @return string
     */
    protected function getReferencedEntityType(string $fieldName): string
    {
        $fieldDefinitions = \Drupal::entityTypeManager()->getDefinition('user', 'user');

        if ($fieldDefinitions->hasKey($fieldName)) {
            $fieldDefinition = $fieldDefinitions->get($fieldName);
            if ($fieldDefinition->getSetting('target_type')) {
                return $fieldDefinition->getSetting('target_type');
            }
        }
        return 'node';
    }

    private function cleanEntity(mixed $entity): mixed
    {

        if (
            $entity instanceof Comment &&
            $entity->hasField('subject') &&
            !mb_check_encoding($entity->get('subject')->value, 'UTF-8')
        )
        {
            $subject = $entity->get('subject')->value;
            $subject = html_entity_decode($subject, ENT_QUOTES | ENT_HTML5, 'UTF-8');
            $subject = mb_convert_encoding($subject, 'UTF-8', 'UTF-8');
            $subject = preg_replace('/[^\x{0000}-\x{10FFFF}]/u', '', $subject);
            $entity->set('subject', $subject);
        }

        return $entity;
    }
}
