<?php
declare(strict_types=1);

namespace Inside\Host\Bridge;

use Drupal;
use Drupal\Core\Session\AnonymousUserSession;
use Drupal\user\Entity\User;
use Inside\Authentication\Models\User as InsideUser;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpFoundation\Session\SessionInterface;

/**
 * Bridge for authentication
 *
 * @category  Class
 * @package   Inside\Host\Bridge\BridgeAuthentication
 * @author    Maecia <technique@maecia.com>
 * @link      http://www.maecia.com/
 */
final class BridgeAuthentication extends Bridge
{
    /**
     *
     * BridgeAuthentication constructor.
     */
    public function __construct()
    {
        parent::__construct(false);
    }

    /**
     * Check if user is logged in
     */
    public function isLoggedIn(?InsideUser $user): bool
    {
        $uuid = $this->current();

        if (!is_null($user) && $uuid == $user->uuid_host) {
            return true;
        }

        return false;
    }

    /**
     * Log the given user to Drupal
     */
    public function loginUser(InsideUser $user): void
    {
        $drupalUser = Drupal::service('entity.repository')->loadEntityByUuid('user', $user->uuid_host);

        if ($drupalUser) {
            user_login_finalize($drupalUser);
            // We force session to be saved
            /** @var SessionInterface $session */
            $session = Drupal::service('session');
            $session->save();
        }
    }

    /**
     * Logout the given user from Drupal
     */
    public function logoutUser(InsideUser $user): void
    {
        if (Auth::check() && Auth::user()->uuid === $user->uuid) {
            // Logout current drupal session
            user_logout();

            return;
        }

        $drupalUser = Drupal::service('entity.repository')->loadEntityByUuid('user', $user->uuid_host);

        if ($drupalUser) {
            $currentUserSession = Drupal::currentUser();
            // Change currentUser to user that needs to logout
            Drupal::currentUser()->setAccount($drupalUser);
            $drupalUserSession = Drupal::currentUser();
            if ($drupalUserSession->isAuthenticated()) {
                Drupal::logger('user')->notice(
                    'Session closed for %name.',
                    ['%name' => $drupalUserSession->getAccountName()]
                );
                Drupal::moduleHandler()->invokeAll('user_logout', [$drupalUserSession]);
                $session_manager = Drupal::service('session_manager');
                $session_manager->delete($drupalUserSession->id());
                $drupalUserSession->setAccount(new AnonymousUserSession());
            }
            Drupal::currentUser()->setAccount($currentUserSession);
        }
    }

    /**
     * Check if the current user is authenticated
     */
    public function isAuthenticated($user = false): bool
    {
        if (!$user) {
            $user = Drupal::currentUser();
        }

        return $user->isAuthenticated();
    }

    /**
     * Check if the current user is authenticated
     */
    public function current(): string|null
    {
        $currentUser = Drupal::currentUser();

        if (!$this->isAuthenticated($currentUser)) {
            return null;
        }

        $drupalUser = User::load($currentUser->id());

        return $drupalUser?->uuid();
    }
}
