<?php

use Drupal\Core\Entity\EntityInterface;
use Inside\Support\Str;

/**
 * Return the drupal entity bundle name
 *
 * @param EntityInterface $entity
 * @return string
 */
function get_entity_bundle(EntityInterface $entity): ?string
{
    switch (get_class($entity)) {
        case 'Drupal\paragraphs\Entity\ParagraphsType':
            return $entity->get('id');
        case 'Drupal\comment\Entity\CommentType':
            return $entity->get('id');
        case 'Drupal\node\Entity\NodeType':
            return $entity->get('type');
        case 'Drupal\node\Entity\Node':
        case 'Drupal\paragraphs\Entity\Paragraph':
            return $entity->getType();
        case 'Drupal\comment\Entity\Comment':
            return $entity->getTypeId();
        case 'Drupal\menu_item_extras\Entity\MenuItemExtrasMenuLinkContent':
        case 'Drupal\menu_link_content\Entity\MenuLinkContent':
            return $entity->getMenuName();
        case 'Drupal\user\Entity\User':
            return 'user';
    }

    return null;
}

/**
 * Return the lumen entity bundle name
 *
 * @param EntityInterface $entity
 * @return string
 */
function get_lumen_entity_bundle(EntityInterface $entity): ?string
{
    $bundle = get_entity_bundle($entity);

    if (!$bundle) {
        return null;
    }

    return match (get_class($entity)) {
        'Drupal\menu_item_extras\Entity\MenuItemExtrasMenuLinkContent',
        'Drupal\menu_link_content\Entity\MenuLinkContent' => Str::studly($bundle.'_menus'),
        'Drupal\user\Entity\User' => Str::studly('users'),
        default => Str::studly($bundle),
    };
}

/**
 * Get the lumen entity model name
 */
function get_lumen_entity_model_name(?EntityInterface $entity = null): ?string
{
    if (is_null($entity)) {
        return null;
    }

    $domain = 'Contents';
    $model = get_lumen_entity_bundle($entity);

    if (is_null($model)) {
        return null;
    }

    if (in_array(
        get_class($entity),
        ['Drupal\paragraphs\Entity\Paragraph', 'Drupal\paragraphs\Entity\ParagraphsType']
    )
    ) {
        $domain = 'Sections';
    }

    return 'Inside\Content\Models\\'.$domain.'\\'.$model;
}

/**
 * Return the lumen entity uuid
 *
 * @param EntityInterface $entity
 * @param string|null                         $langcode langcode of the needed lumen model from entity
 * @return string|null
 */
function get_lumen_entity_uuid(EntityInterface $entity, ?string $langcode = null): ?string
{
    try {
        $content = get_lumen_entity($entity, $langcode);
    } catch (Exception) {
        return null;
    }

    if ($content) {
        if (!isset($content->uuid)) {
            return $content->id;
        }

        return $content->uuid;
    }

    return null;
}

/**
 * Return the lumen entity
 *
 * @param EntityInterface $entity
 * @param string|null                         $langcode langcode of the needed lumen model from entity
 * @return \Illuminate\Database\Eloquent\Model|null
 */
function get_lumen_entity(EntityInterface $entity, ?string $langcode = null)
{
    Drupal::service('inside');

    $model = get_lumen_entity_model_name($entity);

    if ($model === null) {
        return null;
    }

    $query = call_user_func($model . '::withoutGlobalScopes');

    $query->where('uuid_host', $entity->uuid());

    if (!$langcode) {
        $langcode = $entity->language()->getId();
    }

    if (!($entity instanceof \Drupal\user\Entity\User)) {
        $query->where('langcode', $langcode);
    }

    return $query->first();
}

/**
 * Try to guess drupal entity type with given lumen model
 *
 * @param string $type inside type or class name
 * @return string|null
 */
function guess_drupal_entity_type(string $type): string
{
    $drupalTypes = ['node', 'menu_link_content', 'comment', 'user', 'paragraph'];
    if (Str::startsWith($type, 'Inside\\Content\\Models\\')) {
        // Wrong use of guess_drupal_entity_type
        $type = class_to_type($type);
    }

    foreach ($drupalTypes as $drupalType) {
        $bundles = array_keys(Drupal::service('entity_type.bundle.info')->getBundleInfo($drupalType));

        if ($drupalType == 'user') {
            $bundles = ['users'];
        }

        if ($drupalType == 'menu_link_content') {
            // Remove _menus suffix
            $type = str_replace('_menus', '', $type);
        }

        if (in_array(Str::snake($type), $bundles)) {
            return $drupalType;
        }
    }

    return 'node';
}

/**
 * Try to guess drupal entity type with given lumen model
 *
 * @return string $type
 */
function guess_drupal_entity_class(string $type): ?string
{
    switch ($type) {
        case 'node':
            return \Drupal\node\Entity\Node::class;
        case 'paragraph':
            return \Drupal\paragraphs\Entity\Paragraph::class;
        case 'comment':
            return \Drupal\comment\Entity\Comment::class;
        case 'user':
            return \Drupal\user\Entity\User::class;
        case 'menu_link_content':
            return \Drupal\menu_link_content\Entity\MenuLinkContent::class;
    }

    return null;
}

/**
 * Return the drupal entity uuid
 *
 * @param string $uuid
 * @return string|null
 */
function get_drupal_entity_uuid(string $uuid): ?string
{
    $content = call_user_func('searchModel::find', $uuid);

    if ($content) {
        return $content->uuid_host;
    }

    return null;
}

/**
 * Return field name for a node bundle
 */
function get_drupal_paragraph_fields(string $domain, string $bundle): array
{
    $fields = Drupal::service('entity_field.manager')->getFieldDefinitions($domain, $bundle);

    $paragraphFields = [];
    foreach ($fields as $definition) {
        if (!empty($definition->getTargetBundle()) && $definition->getType() == 'entity_reference_revisions') {
            $paragraphFields[] = $definition->getName();
        }
    }

    return $paragraphFields;
}

/**
 * Return the entity type by parent type
 */
function get_drupal_children_type(string $type): ?string
{
    return match ($type) {
        'node_type' => 'node',
        'comment_type' => 'comment',
        'paragraphs_type' => 'paragraph',
        'menu' => 'menu_link_content',
        default => null,
    };
}

/**
 * Get a third party setting for an entity
 *
 * @param      $entity
 * @param      $module
 * @param      $key
 * @param bool $subKey
 * @return mixed
 */
function get_third_party_setting($entity, $module, $key, $subKey = false)
{
    $setting = $entity->getThirdPartySetting($module, $key);

    if (!$setting && $subKey) {
        $subSetting = $entity->getThirdPartySetting($module, $subKey);

        if ($subSetting) {
            return $subSetting;
        }
    }

    if (!$subKey) {
        return $setting;
    }
    if (is_array($setting) && array_key_exists($subKey, $setting)) {
        return $setting[$subKey];
    }

    return null;
}

/**
 * Get a third party setting for an entity
 *
 * @param       $entity
 * @param       $config
 * @param       $module
 * @param       $key
 * @param bool  $subKey
 * @param mixed $default
 * @return mixed
 */
function get_setting($entity, $config, $module, $key, $subKey = false, $default = '')
{
    return ($entity
            ? get_third_party_setting($entity, $module, $key, $subKey)
            : $config->get(
                $subKey !== false ? $subKey : $key
            )) ?? $default;
}
