<?php

namespace Inside\Host\Normalizer\Field;

use Drupal\Core\Url;
use Drupal\link\Plugin\Field\FieldType\LinkItem;
use Drupal\node\Entity\Node;
use Drupal\serialization\Normalizer\NormalizerBase;
use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Inside\Slug\Models\Slug;
use Symfony\Component\Serializer\Normalizer\DenormalizerInterface;
use Symfony\Component\Serializer\Normalizer\NormalizerInterface;

/**
 * Link Normalizer
 *
 * @category  Class
 * @package   Inside\Host\Normalizer\Field\LinkNormalizer
 * @author    Maecia <technique@maecia.com>
 * @copyright 2018 Maecia
 * @link      http://www.maecia.com/
 */
class LinkNormalizer extends NormalizerBase implements DenormalizerInterface, NormalizerInterface
{
    /**
     * {@inheritDoc}
     */
    protected $supportedInterfaceOrClass = LinkItem::class;

    /**
     * {@inheritDoc}
     */
    public function normalize($object, $format = null, array $context = [])
    {
        $values = $object->getValue();
        $uri    = $values['uri'] ?? '';

        if (inside_link_is_external($uri)) {
            return $uri;
        }

        $path = explode(':', $uri);

        if (in_array($path[1], ['#', '<front>', '/'])) {
            return '';
        }

        $info = parse_url($uri);
        $url = (array_key_exists('scheme', $info) && in_array($info['scheme'], ['http', 'https'])) ? $path[1] : '/' . $path[1];

        $parameters = Url::fromUserInput($url);

        if (!$parameters->isRouted()) {
            if (strstr($path[1], 'node/')) {
                $parameters = [
                    'node' => str_replace('node/', '', $path[1]),
                ];
            } else {
                return '';
            }
        } else {
            $parameters = $parameters->getRouteParameters();
        }

        if (isset($parameters['node'])) {
            $entity = Node::load($parameters['node']);
        } else {
            [$type, $id] = explode("/", $path[1]);
            $entity = \Drupal::entityTypeManager()->getStorage($type)->load($id);
        }

        if (!$entity) {
            return '';
        }

        try {
            $langcode = null;
            if (isset($context['langcode'])) {
                $langcode = $context['langcode'];
            }
            $content = get_lumen_entity($entity, $langcode);

            if (isset($content->slug) && !empty($content->slug)) {
                return $content->slug[0];
            } else {
                return 'uuid:' . $content->content_type . ':' . $content->uuid;
            }
        } catch (\Exception $e) {
            Log::error('[LinkNormalizer] failed to normalize [' . $e->getMessage() . ']');
        }

        return 'home'; // Link not found redirect to home
    }

    /**
     * {@inheritDoc}
     */
    public function denormalize($data, $class, $format = null, array $context = [])
    {
        \Drupal::service('inside');

        if (!isset($context['target_instance'])) {
            throw new \InvalidArgumentException(
                '$context[\'target_instance\'] must be set to denormalize with the FieldNormalizer'
            );
        }

        if ($context['target_instance']->getParent() == null) {
            throw new \InvalidArgumentException(
                'The field passed in via $context[\'target_instance\'] must have a parent set.'
            );
        }

        $items = $context['target_instance'];

        if (!inside_link_is_external($data)) {
            // On internal link, extract path from uri
            $data = str_replace(config('app.url'), '', $data);
        }

        try {
            $slug = Slug::where('slug', trim($data, '/'))->firstOrFail();

            $query = call_user_func(
                'Inside\\Content\\Models\\Contents\\' . Str::studly(Str::substr($slug->type, 15)) . '::query'
            );
            $item  = $query->findOrFail($slug->uuid);

            $node = \Drupal::service('entity.repository')->loadEntityByUuid('node', $item->uuid_host);

            $items->setValue(['uri' => 'entity:node/' . $node->id(), 'title' => null, 'options' => null]);
        } catch (Exception $e) {
            // We did not find a valid link attached to a node
            if (inside_link_is_external($data)) {
                $items->setValue(['uri' => $data, 'title' => null, 'options' => null]);
            }
        }
    }
}
