<?php

namespace Inside\Host\Normalizer\Entity;

use Drupal\user\Entity\User;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;
use Inside\Permission\Models\Role;

/**
 * User Normalizer
 *
 * @category  Class
 * @package   Inside\Host\Normalizer\Entity\UserNormalizer
 * @author    Maecia <technique@maecia.com>
 * @author    Nicolas Deniaud <contact@feldoe.net>
 * @copyright 2018 Maecia
 * @link      http://www.maecia.com/
 */
class UserNormalizer extends BaseEntityNormalizer
{

    /**
     * {@inheritdoc}
     */
    protected $supportedInterfaceOrClass = \Drupal\user\UserInterface::class;

    /**
     * {@inheritdoc}
     */
    protected $removableFields = [
        'uid',
        'preferred_langcode',
        'preferred_admin_langcode',
        'timezone',
        'access',
        'login',
        'init',
        'default_langcode',
        'content_translation_source',
        'content_translation_outdated',
        'content_translation_uid',
        'content_translation_status',
        'content_translation_created',
        'roles',
        //These fields does only get value inside side
        'field_level',
        'field_score',
        'field_statistics',
    ];

    /**
     * {@inheritdoc}
     */
    protected $swappableNames = [
        'mail' => 'email',
        'pass' => 'password',
        'created' => 'created_at',
        'changed' => 'updated_at',
        'uuid' => 'uuid_host',
    ];

    /**
     * {@inheritDoc}
     */
    public function normalize($entity, $format = null, array $context = [])
    {
        if (!isset($context['langcode'])) {
            $context['langcode'] = $entity->language()->getId();
        }

        $attributes = parent::normalize($entity, $format, $context);

        if (array_key_exists('content_translation_uid', $attributes)
            && !is_null($authorId = Arr::first($attributes['content_translation_uid']))) {
            $attributes['author'] = is_int($authorId) ?
                $this->getUserUuid($authorId) :
                $authorId;
        } else {
            $superAdminRole = Role::where('name', 'super_administrator')->first();
            if ($superAdminRole) {
                $attributes['author'] = $superAdminRole->users()->first()->uuid ?? null;
            }
        }
        if (is_null($attributes['author'])) {
            $attributes['author'] = \Inside\Authentication\Models\User::first()->uuid;
        }

        $normalized = $this->formatDatasToNormalization($entity, $attributes);

        if ($normalized['password'] === $entity->get('pass')->value) {
            unset($normalized['password']);
        }

        return $normalized;
    }

    /**
     * {@inheritDoc}
     */
    public function denormalize($data, $class, $format = null, array $context = [])
    {
        $values = [];
        $data = $this->prepareLangs($data, $values, 'default_langcode', 'langcode');

        $toRemove = ['bundle', 'type', 'roles'];

        foreach ($toRemove as $field) {
            if (isset($data[$field])) {
                unset($data[$field]);
            }
        }

        if (isset($data['langcode'])) {
            $data['preferred_langcode'] = $data['langcode'];
            $data['preferred_admin_langcode'] = $data['langcode'];
        }
        $entity = null;
        try {
            $entity = $this->createEntity($values, $data, $context, 'user', 'uid');
            $result = $this->formatDatasToDenormalization($data, 'user', 'user');

            unset($result['author']);
            $this->denormalizeFieldData($result, $entity, $format, $context);
        } catch (\Exception $e) {
            Log::error('[UserNormalizer] User denormalization failed ['.$e->getMessage().']');
            Log::debug('[UserNormalizer] User denormalization failed trace ['.$e->getTraceAsString().']');
        }

        return $entity;
    }

    protected function getUserUuid(int $uid): ?string
    {
        $author = User::load($uid);
        if ($author) {
            $insideUser = \Inside\Authentication\Models\User::where('uuid_host', $author->uuid())->first();
            if (is_null($insideUser)) {
                return null;
            }
            return $insideUser->uuid;
        }
        return null;
    }
}
